#
# Cookbook:: app_central_voice
# Recipe:: default
#
# Copyright:: 2017, Twitch, All Rights Reserved.

set_role_tag
include_recipe 'base_server_windows'
app_pool = 'CurseVoice'

# Install IIS and .NET v4.5
include_recipe 'iis'
include_recipe 'iis::mod_aspnet45'
include_recipe 'iis::mod_compress_dynamic'

windows_feature 'Web-Mgmt-Service' do
  install_method :windows_feature_powershell
  action :install
end

service 'WMSVC' do
  action [:enable, :start]
end

# Enables remote IIS management
registry_key 'HKEY_LOCAL_MACHINE\Software\Microsoft\WebManagement\Server' do
  values [{
              :name => 'EnableRemoteManagement',
              :type => :dword,
              :data => 1
          }]
  notifies :restart, 'service[WMSVC]', :immediately
end

powershell_script 'DynamicCompression JSON' do
  code 'add-webconfiguration "/system.webServer/httpCompression/dynamicTypes" -pspath IIS:\ -value (@{mimeType="application/json";enabled="true"})'
  not_if '$test = Get-webconfiguration -filter "/system.webServer/httpCompression/dynamicTypes/add[@mimeType=\'application/json\']" -pspath IIS:; if ($test -eq $null) { exit(1) }'
end

powershell_script 'DynamicCompression JSON UTF-8' do
  code 'add-webconfiguration "/system.webServer/httpCompression/dynamicTypes" -pspath IIS:\ -value (@{mimeType="application/json; charset=utf-8";enabled="true"})'
  not_if '$test = Get-webconfiguration -filter "/system.webServer/httpCompression/dynamicTypes/add[@mimeType=\'application/json; charset=utf-8\']" -pspath IIS:; if ($test -eq $null) { exit(1) }'
end

powershell_script 'Remove X-Powered-By' do
  code 'Clear-WebConfiguration "/system.webServer/httpProtocol/customHeaders/add[@name=\'X-Powered-By\']"'
  only_if '$test = Get-WebConfiguration system.webServer/httpProtocol/customHeaders/* | where { $_.name -eq \'X-Powered-By\' }; if ($test -eq $null) { exit(1) }'
end

powershell_script 'Increase Connection Limits' do
  code 'Set-WebConfiguration -Filter "/system.webserver/serverRuntime" -PsPath IIS:\ -Value (@{appConcurrentRequestLimit=50000})'
  not_if '(Get-WebConfigurationProperty -Filter "/system.webserver/serverRuntime" -PSPath IIS:\ -Name appConcurrentRequestLimit).Value -eq 50000'
end

powershell_script 'Disable Request Logging' do
  code 'Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST" -filter "system.webServer/httpLogging" -name "dontLog" -value "True"'
  not_if '(Get-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST" -filter "system.webServer/httpLogging" -name "dontLog").Value -eq "True"'
end

# Remove default App Pools and Sites
iis_site 'Default Web Site' do
  action [:stop, :delete]
end

['DefaultAppPool', '.NET v4.5', '.NET v4.5 Classic'].each do |pool|
  iis_pool pool do
    action [:stop, :delete]
  end
end


# Create HealthMonitor Pool and Site
directory "#{node['iis']['pubroot']}/HealthMonitor" do
  action :create
end

iis_pool 'HealthMonitor' do
  runtime_version '4.0' # 4.0 IS REALLY 4.5
  start_mode :AlwaysRunning
  identity_type :LocalSystem
  rapid_fail_protection false
  idle_timeout '00:00:00'
  recycle_after_time "00:00:00"
  disallow_rotation_on_config_change true
  action [:add, :config]
end

iis_site 'HealthMonitor' do
  bindings 'http/*:80:'
  path "#{node['iis']['pubroot']}/HealthMonitor"
  application_pool 'HealthMonitor'
  action [:add, :config]
end

directory '/opt/scripts' do
  recursive true
end

directory "/opt/packages" do
  action :create
end

cookbook_file "/opt/packages/healthmonitor_current.zip" do
  source 'healthmonitor_current.zip'
end

cookbook_file "/opt/scripts/Deploy-HealthMonitor.ps1" do
  source 'Deploy-HealthMonitor.ps1'
end

cookbook_file '/Windows/Microsoft.NET/Framework64/v4.0.30319/aspnet.config' do
  source 'aspnet.config'
end

cookbook_file '/opt/scripts/Recycle-AppPool.ps1' do
  source 'Recycle-AppPool.ps1'
end

powershell_script "Deploy-HealthMonitor" do
  code 'powershell /opt/scripts/Deploy-HealthMonitor.ps1'
  creates 'c:\inetpub\HealthMonitor\.chef_deployed'
end

windows_feature 'NET-WCF-HTTP-Activation45' do
  install_method :windows_feature_powershell
  action :install
end

directory "#{node['iis']['pubroot']}/#{app_pool}/Current" do
  recursive true
  action :create
end

iis_pool app_pool do
  runtime_version '4.0' # 4.0 IS REALLY 4.5
  start_mode :AlwaysRunning
  identity_type :LocalSystem
  rapid_fail_protection false
  idle_timeout '00:00:00'
  recycle_after_time "00:00:00"
  disallow_rotation_on_config_change true
  disallow_overlapping_rotation true
  action [:add, :config]
end

iis_site app_pool do
  bindings node['iis']['bindings'].to_a.join(',')
  path "#{node['iis']['pubroot']}/#{app_pool}/Current"
  application_pool app_pool
  action [:add, :config]
end

windows_task 'RecycleAppPool' do
  user 'SYSTEM'
  command "c:\\windows\\system32\\WindowsPowerShell\\v1.0\\powershell.exe -ExecutionPolicy Bypass -Command c:\\opt\\scripts\\Recycle-AppPool.ps1 -AppPool #{app_pool}"
  run_level :highest
  frequency :daily
  start_time sprintf('%02d:%02d', node['iis']['recycle_times'][get_availability_zone.to_sym], fqdn_rand(60))
end