#!/usr/bin/swift

import Foundation

let clientProfileFrameworks = [
    "Accessibility",
    "CustomMarshalers",
    "Microsoft.CSharp",
    "Microsoft.JScript",
    "Microsoft.VisualBasic.Compatibility.Data",
    "Microsoft.VisualBasic.Compatibility",
    "Microsoft.VisualBasic",
    "Microsoft.VisualC",
    "mscorlib",
    "PresentationCore",
    "PresentationFramework.Aero",
    "PresentationFramework.Classic",
    "PresentationFramework",
    "PresentationFramework.Luna",
    "PresentationFramework.Royale",
    "ReachFramework",
    "sysglobl",
    "System.Activities.Core.Presentation",
    "System.Activities",
    "System.Activities.DurableInstancing",
    "System.Activities.Presentation",
    "System.Addin.Contract",
    "System.Addin",
    "System.ComponentModel.Composition",
    "System.ComponentModel.DataAnnotations",
    "System.configuration",
    "System.Configuration.Install",
    "System.Core",
    "System.Data.DataSetExtensions",
    "System.Data",
    "System.Data.Entity",
    "System.Data.Linq",
    "System.Data.Services.Client",
    "System.Data.SqlXml",
    "System.Deployment",
    "System.Device",
    "System.DirectoryServices.AccountManagement",
    "System.DirectoryServices",
    "System.DirectoryServices.Protocols",
    "System",
    "System.Drawing",
    "System.EnterpriseServices",
    "System.EnterpriseServices.Thunk",
    "System.EnterpriseServices.Wrapper",
    "System.IdentityModel",
    "System.IdentityModel.Selectors",
    "System.IO.Log",
    "System.Management",
    "System.Management.Instrumentation",
    "System.Messaging",
    "System.Net",
    "System.Numerics",
    "System.Printing",
    "System.Runtime.DurableInstancing",
    "System.Runtime.Remoting",
    "System.Runtime.Serialization",
    "System.Runtime.Serialization.Formatters.Soap",
    "System.Security",
    "System.ServiceModel.Activities",
    "System.ServiceModel.Channels",
    "System.ServiceModel.Discovery",
    "System.ServiceModel",
    "System.ServiceModel.Routing",
    "System.ServiceProcess",
    "System.Speech",
    "System.Transactions",
    "System.Web.ApplicationServices",
    "System.Web.Services",
    "System.Windows.Forms.DataVisualization",
    "System.Windows.Forms",
    "System.Windows.Input.Manipulations",
    "System.Windows.Presentation",
    "System.Xaml",
    "System.Xml",
    "System.Xml.Linq",
    "UIAutomationClient",
    "UIAutomationClientsideProviders",
    "UIAutomationProvider",
    "UIAutomationTypes",
    "WindowsBase",
    "WindowsFormsIntegration",
    
    "Mono.Posix"
]

func shell(launchPath: String, arguments: String...) -> String
{
    let process = Process()
    process.launchPath = launchPath
    process.arguments = arguments
    
    let pipe = Pipe()
    process.standardOutput = pipe
    process.launch()
    
    let data = pipe.fileHandleForReading.readDataToEndOfFile()
    let output = String(data: data, encoding: .utf8)!
    
    return output
}

func assemblyName(element:String) -> String? {
    
    if let match = element.range(of: "(?<=AssemblyName=\")[^\"]+(?=\")", options: .regularExpression) {
        return String(element[match])
    }
    else {
        return nil
    }
}

func assemblyPublicKeyToken(path:String, sn:String) -> String? {
    
    let ret = shell(launchPath: sn, arguments: "-T", path)
    
    if let match = ret.range(of: "(?<=Public Key Token: )[^\n]+(?=\n)", options: .regularExpression) {
        return String(ret[match])
    }
    else {
        return nil
    }
}

let fileManager = FileManager.default

let monoDir = "/Library/Frameworks/Mono.framework/Versions/Current"
let monoLibDir = monoDir + "/lib/mono"
let snPath = monoDir + "/bin/sn"
let frameworkListFilePath = monoDir + "/lib/mono/xbuild-frameworks/.NETFramework/v4.0/Profile/Client/RedistList/FrameworkList.xml"
let frameworkTargetDir = monoDir + "/lib/mono/4.0-api"

guard nil == [monoDir, snPath, frameworkTargetDir, frameworkListFilePath].first(where: { !fileManager.fileExists(atPath: $0) })
    , let enumerator = fileManager.enumerator(at: URL(fileURLWithPath: monoLibDir), includingPropertiesForKeys: nil) else {
    print("Mono is not properly installed.")
    exit(0)
}

func patch(url:URL, replace target:String, with replacement:String) throws {
    
    var contents = try String(contentsOf: url)

    if let range = contents.range(of: target) {
        
        contents.replaceSubrange(range, with: replacement)
        
        try contents.write(to: url, atomically: true, encoding: .utf8)
        
        print("Patched: \(url)\nreplaced: \(target)\nwith: \(replacement)\n")
    }
}

do {
    try enumerator.forEach { (url) in
        
        guard let url = url as? URL
            , url.pathExtension == "targets" else { return }
        
        if url.lastPathComponent.hasPrefix("Microsoft.CSharp.") {
            
            try patch(url: url,
                      replace: "DefineConstants=\"$(DefineConstants)\"",
                      with: "DefineConstants=\"$(DefineConstants);__MonoCS__\"")
        }
        else if url.lastPathComponent == "Microsoft.Common.CurrentVersion.targets" {
            
            try patch(url: url,
                      replace: "Condition=\"'@(COMReference)'!='' or '@(COMFileReference)'!=''\"",
                      with: "Condition=\"'False'\"")
        }
    }
}
catch _ {
    print("Failed to define constants. Make sure you run the app as sudo.");
    exit(0)
}

var contents:[String]

do {
    contents = try String(contentsOfFile: frameworkListFilePath, encoding:.utf8).components(separatedBy: "\n");
}
catch _ {
    print("Failed to open framework list file");
    exit(0)
}

let includedAssemblies = Set(contents.flatMap(assemblyName))
let missingAssemblies = Set(clientProfileFrameworks).subtracting(includedAssemblies)

let formattedElements = missingAssemblies.flatMap { assemblyName -> String? in
    
    let path = frameworkTargetDir + "/" + assemblyName + ".dll"
    
    guard FileManager.default.fileExists(atPath: path)
        , let publicKeyToken = assemblyPublicKeyToken(path: path, sn: snPath) else { return nil }
    
    return "<File AssemblyName=\"" + assemblyName + "\" Version=\"4.0.0.0\" PublicKeyToken=\"" + publicKeyToken + "\" />"
}

let insertionIndex = contents.index(of: "</FileList>")!

contents.insert(contentsOf: formattedElements, at: insertionIndex);

let fixed = contents.joined(separator: "\n");

do {
    try fixed.write(toFile: frameworkListFilePath, atomically: true, encoding: .utf8)
}
catch let error {
    print("Failed to write output file: \(error)")
}


