require 'ffi'
require 'json'

module Sandstorm
  module Go
    class String < FFI::Struct
      layout \
        :p,     :pointer,
        :len,   :long_long
    end

    # Convert a ruby string to a Go string.
    def Go.StringFromRubyString(value)
      res = Go::String.new
      res[:p] = FFI::MemoryPointer.from_string(value)
      res[:len] = value.size
      return res
    end

    # Return a ruby string and free the C value of a string.
    def Go.ExtractString(p)
      value = p.read_string()
      LibC.free(p)
      return value
    end

    # Check a pointer for an error and raise it if not nil.
    def Go.CheckError(sp)
      if sp == nil
        return
      end
      raise Go.ExtractString(sp)
    end

    def Go.Call(func, input)
      res = func.call(Go.StringFromRubyString(JSON.generate(input)))
      Go.CheckError(res[:r1])
      return JSON.parse(Go.ExtractString(res[:r0]))
    end

    class Response < FFI::Struct
      layout \
        :r0,  :pointer,
        :r1,  :pointer
    end
  end

  module LibC
    extend FFI::Library
    ffi_lib FFI::Library::LIBC

    attach_function :free, [:pointer], :void
  end

  module LibSandstorm
    extend FFI::Library

    begin
      # try loading installed lib
      ffi_lib 'libsandstorm.so'
    rescue LoadError
      # try loading packaged lib
      ffi_lib File.join(__dir__, 'libsandstorm.so')
    end

    attach_function :create_secret, [Go::String.by_value], Go::Response.by_value
    attach_function :get_secret, [Go::String.by_value], Go::Response.by_value
    attach_function :update_secret_plaintext, [Go::String.by_value], Go::Response.by_value
    attach_function :delete_secret, [Go::String.by_value], Go::Response.by_value
  end

  class Client
    def initialize(region: 'us-west-2', role_arn: '', environment: '', access_key: '', secret_key: '')
      @region = region
      @role_arn = role_arn
      @environment = environment
      @access_key = access_key
      @secret_key = secret_key
    end

    def call(fn, input)
      return Go.Call(fn, input.merge({
        :Region => @region,
        :Environment => @environment,
        :RoleARN => @role_arn,
        :AccessKeyID => @access_key,
        :SecretAccessKey => @secret_key,
        :SecurityToken => ENV.fetch('AWS_SECURITY_TOKEN', ''),
        :SessionToken => ENV.fetch('AWS_SESSION_TOKEN', ''),
      }))
    end

    # Create a secret.
    def create_secret(secret_name, secret_plaintext, secret_class='')
      return self.call(LibSandstorm.method(:create_secret), {
        :Name => secret_name,
        :Plaintext => secret_plaintext,
        :Class => secret_class,
      })
    end

    # Get a secret's plaintext value.
    def get_secret_plaintext(secret_name)
      return self.call(LibSandstorm.method(:get_secret), {
        :Name => secret_name,
      })['Plaintext']
    end

    # Update a secret's plaintext value.
    def update_secret_plaintext(secret_name, secret_plaintext)
      return self.call(LibSandstorm.method(:update_secret_plaintext), {
        :Name => secret_name,
        :Plaintext => secret_plaintext,
      })
    end

    # Delete a secret.
    def delete_secret(secret_name)
      return self.call(LibSandstorm.method(:delete_secret), {
        :Name => secret_name,
      })
    end
  end
end

Chef::Recipe.send(:include, Sandstorm)
Chef::Provider.send(:include, Sandstorm)
Chef::Resource.send(:include, Sandstorm)
Chef::Mixin::Template::TemplateContext.send(:include, Sandstorm)
Chef::Node::Attribute.send(:include, Sandstorm)