#! /usr/bin/env ruby
# frozen_string_literal: true

require 'sensu/extension/tueor'
require 'tempfile'

module Sensu::Extension
  class DiskReadOnlyCheck < TueorCheck

    def name
      'linux-disk-readonly-check'
    end

    def description
      'Check for read-only filesystem on a linux machine'
    end

    def definition
      {
          :type => "check",
          :name => name
      }
    end

    prep_step(:set_defaults) do
      set :folder, '/'
    end

    step(:write_tempfile, :is_critical, :message) do
      operation = proc do
        f = nil
        begin
          f = Tempfile.new('readonly_check', folder)
          f.write('.')
          set :is_critical, false
          set :message, "wrote to test file"
        rescue => e
          set :is_critical, true
          set :message, "Failed to write out test file: #{e}"
        ensure
          f.close if f
          f.unlink if f
        end
      end

      EM.defer(operation)
    end

    def critical?
      is_critical
    end

    def warning?
      critical?
    end

  end
end