#! /usr/bin/env ruby
# frozen_string_literal: true

require 'sensu/extension/tueor'

module Sensu::Extension
  class DiskSpaceCheck < TueorCheck

    def name
      'linux-diskspace-check'
    end

    def description
      'Check the disk space and inode usage on a linux machine'
    end

    def definition
      {
          :type => "check",
          :name => name
      }
    end

    prep_step(:set_default_thresholds) do
      set :critical_threshold, 90
      set :warning_threshold, 85
      set :mountpoint, '/'
    end

    step(:get_disk_usage, :blocks_used, :blocks_utilization, :inodes_used, :inodes_utilization, :message) do
      command(:df).arg(:lTP, short: true) do |output|
        blocks_used, blocks_utilization = parse(output, mountpoint)
        set :blocks_used, blocks_used
        set :blocks_utilization, blocks_utilization

        msg_part1 = "#{mountpoint}    #{blocks_utilization}%  #{blocks_used}KB Used"

        command(:df).arg(:ilTP, short: true) do |output|
          inodes_used, inodes_utilization = parse(output, mountpoint)
          set :inodes_used, inodes_used
          set :inodes_utilization, inodes_utilization

          set :message, "#{msg_part1}    #{inodes_utilization}%  #{inodes_used} Inodes Used"
        end
      end
    end

    def parse(data, mount)
      data = data.split("\n").select{ |line| line.strip.end_with? mount }[0]
      _, _, _, used, _, percentage, _ = data.split(/\s+/)
      return used.to_i, percentage.to_i
    end

    def critical?
      [blocks_utilization, inodes_utilization].max >= critical_threshold
    end

    def warning?
      [blocks_utilization, inodes_utilization].max >= warning_threshold
    end

  end
end