#! /usr/bin/env ruby
# frozen_string_literal: true

require 'sensu/extension/tueor'

module Sensu::Extension
  class MemorySwapCheck < TueorCheck

    def name
      'linux-memoryswap-check'
    end

    def description
      'Check the memory swap usage on a linux machine'
    end

    def definition
      {
          :type => "check",
          :name => name
      }
    end

    prep_step(:set_default_thresholds) do
      set :critical_threshold, 25
      set :warning_threshold, 10
    end

    step(:get_memoryswap_usage, :used_percentage, :message) do
      path = File.join('/proc', 'meminfo')
      read_file(path) do |output|

        stats = Hash.new

        output.split("\n").each do |line|
          stat, value, _ = line.split(/[:\s]+/)
          stats[stat.snakecase] = value.to_f
        end

        if stats['swap_total'] == 0
          set :used_percentage, 0
          set :message, 'Swap Disabled'
        else
          swap_used_kb = stats['swap_total'] - stats['swap_free']
          set :used_percentage, (1 - stats['swap_free'] / stats['swap_total']) * 100
          set :message, "#{used_percentage.round(2)}%  #{swap_used_kb.to_i}KB Used"
        end

      end
    end

    def critical?
      used_percentage >= critical_threshold
    end

    def warning?
      used_percentage >= warning_threshold
    end

  end
end