#! /usr/bin/env ruby
# frozen_string_literal: true

require 'sensu/extension/tueor'

module Sensu::Extension
  class ZombieProcCheck < TueorCheck

    def name
      'linux-zombieprocs-check'
    end

    def description
      'Check for processes in a Z state on linux machines'
    end

    def definition
      {
          :type => "check",
          :name => name
      }
    end

    prep_step(:set_default_thresholds) do
      set :critical_threshold, 15
      set :warning_threshold, 1
    end

    step(:get_zombie_procs, :zombie_procs, :message) do
      command(:ps).params('axwwo user,pid,vsz,rss,pcpu,nlwp,state,etimes,time,command') do |output|

        zombie_procs = output.split("\n").drop(1).map do |line|
          line_to_hash(line, :user, :pid, :vsz, :rss, :pcpu, :thcount, :state, :etimes, :time, :command)
        end.select{|p| p[:state] == 'Z' && p[:etimes].to_i > 60 }

        set :zombie_procs, zombie_procs
        set :message, zombie_procs.length > 0 ?
            zombie_procs.map{|p| "#{p[:user]}:#{p[:command]}(#{p[:etimes]}s) is stuck in Z state"}.join("\n") :
            "No zombie procs"
      end
    end

    def line_to_hash(line, *cols)
      Hash[cols.zip(line.strip.split(/\s+/, cols.size))]
    end

    def critical?
      zombie_procs.length >= critical_threshold
    end

    def warning?
      zombie_procs.length >= warning_threshold
    end

  end
end