#
# Cookbook:: base_server_windows
# Recipe:: default
#
# Copyright:: 2017, Twitch, All Rights Reserved.

Chef.event_handler do
  on :run_completed do
    CPE::Helpers.send_sensu_payload 'chef-client-heartbeat', 'chef-client-heartbeat OK', 0
  end

  on :run_failed do |exception|
    CPE::Helpers.send_sensu_payload 'chef-client-heartbeat', exception.message, 1
  end
end

powershell_script 'Windows Firewall' do
  code 'Set-NetFirewallProfile -All -Enabled False'
  only_if '(Get-NetFirewallProfile | Where Enabled -eq True).count -gt 0'
end

# Enables remote management
registry_key 'HKEY_LOCAL_MACHINE\System\CurrentControlSet\Control\Terminal Server' do
  values [{
              :name => 'fDenyTSConnections',
              :type => :dword,
              :data => 0
          }]
end

registry_key 'HKEY_LOCAL_MACHINE\System\CurrentControlSet\Control\Terminal Server\WinStations\RDP-Tcp' do
  values [{
              :name => 'UserAuthentication',
              :type => :dword,
              :data => 1
          }]
end

# disable automatic windows updates
registry_key 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update' do
  values [{
              :name => 'AUOptions',
              :type => :dword,
              :data => 1
          }]
end

unless node['windows_dns']['forwarder'].nil?
  windows_feature 'DNS' do
    install_method :windows_feature_powershell
    action :install
    ignore_failure true # breaks on warning for no static IP, but does work
  end

  # dns_forwarder_script = "Get-Module DnsServer -ListAvailable\n"
  node['windows_dns']['forwarder'].each do |zone|
    powershell_script "add-dns-forwarder-for-#{zone[:name]}" do
      code "Add-DnsServerConditionalForwarderZone -Name \"#{zone[:name]}\" -MasterServers #{zone[:servers]} -PassThru"
      not_if "Get-DnsServerZone -Name \"#{zone[:name]}\""
    end
  end

  # set localhost as the dnsclientserveraddress on first primary interface
  powershell_script 'set-dns-client-server-address' do
    code 'Set-DnsClientServerAddress -InterfaceIndex $((Get-NetAdapter -Physical)[0].ifIndex) -ServerAddress ("127.0.0.1")'
  end
end

include_recipe '::admins'

unless node['sensu'] && node['sensu']['prevent_install']
  include_recipe '::monitoring'
end

include_recipe '::ssm_agent'

include_recipe 'chocolatey'

windows_path "C:\\ProgramData\\chocolatey\\bin\\" do
  action :add
end

windows_path "C:\\opscode\\chef\\embedded\\bin\\" do
  action :add
end

cookbook_file 'c:/opscode/chef/embedded/bin/unzip.exe' do
  source 'unzip.bin'
end

windows_task 'ChefPeriodicTask' do
  user 'SYSTEM'
  cwd 'C:\opscode\chef\embedded\bin'
  command 'c:\windows\system32\WindowsPowerShell\v1.0\powershell.exe -ExecutionPolicy Bypass -Command C:\opscode\chef\bin\chef-run.ps1'
  run_level :highest
  frequency :hourly
  random_delay '600'
end

service 'chef-client' do
  action [:disable, :stop]
  only_if '(Get-Service | Where Name -eq "chef-client").length -eq 1'
  ignore_failure true
end


# Experimenting with script to assist with advanced user permissions
# i.e. service accounts, prevent interactive logon, etc.
# directory 'c:/cpe/scripts' do
#   owner 'cpe'
#   group 'Administrators'
#   mode 0700
#   recursive true
# end
# cookbook_file 'c:/cpe/scripts/UserRights.ps1' do
#   source 'UserRights.ps1'
# end

## Seems like WinRM just works now?
# # TODO: Make this idempotent
# powershell_script 'winrm-auth' do
# #  code "winrm set winrm/config/service/Auth '@{Basic=\"true\"}'"
#   code 'Set-WSManInstance -ResourceURI winrm/config/service/Auth -ValueSet @{Basic="true"}'
# end

# # TODO: Make this idempotent
# powershell_script 'winrm-unencrypted' do
# #  code "winrm set winrm/config/service '@{AllowUnencrypted=\"false\"}'"
#   code 'Set-WSManInstance -ResourceURI winrm/config/service -ValueSet @{AllowUnencrypted="true"}'
# end

# # TODO: Make this idempotent
# powershell_script 'winrm-winrs' do
# #  code "winrm set winrm/config/winrs '@{MaxMemoryPerShellMB=\"1024\"}'"
#   code 'Set-WSManInstance -ResourceURI winrm/config/winrs -ValueSet @{MaxMemoryPerShellMB="1024"}'
# end

# # TODO: Make this idempotent
# powershell_script 'winrm-winrs-trustedhosts' do
# #  code "winrm set winrm/config/client '@{TrustedHosts=\"*\"}'"
#   code 'Set-WSManInstance -ResourceURI winrm/config/client -ValueSet @{TrustedHosts="*"}'
# end

# powershell_script 'winrm-http-listener' do
#   code <<-EOH
#     New-WSManInstance -ResourceURI winrm/config/Listener -SelectorSet @{Address="*";Transport="HTTP"}
#   EOH
#   not_if 'Get-WSManInstance -ResourceURI winrm/config/Listener -SelectorSet @{Address="*";Transport="HTTP"}'
#   notifies :run, 'powershell_script[restart-winrm]', :delayed
# end

# powershell_script 'restart-winrm' do
#   code <<-EOH
#     Restart-Service -Force -Name WinRM
#   EOH
#   action :nothing
# end
