module CPE
  module Helpers

    def set_role_tag
      role_tag = "#{cookbook_name}#{"-" + recipe_name unless recipe_name == 'default'}"
      Chef::Log.info "Setting role tag to #{role_tag}"
      tag role_tag
    end

    ## Use this to return node attributes in both recipe and attribute contexts
    def node_helper
      is_a?(Chef::Node::Attribute) ? self : node
    end

    ## Common EC2 Metadata
    def get_region
      node_helper['ec2'].nil? ? '' : node_helper['ec2']['region']
    end

    def get_availability_zone
      node_helper['ec2'].nil? ? '' : node_helper['ec2']['availability_zone'][-1]
    end

    def get_instance_id
      node_helper['ec2'].nil? ? '' : node_helper['ec2']['instance_id']
    end

    def get_aws_account
      node_helper['ec2'].nil? ? '' : node_helper['ec2']['account_id']
    end

    def fqdn_rand(mod = 24, hostname: nil)
      require "digest/md5"
      Digest::MD5.hexdigest(hostname || node_helper['hostname']).hex % mod
    end

    def shadow?
      node.chef_environment.include?('-shadow-')
    end

    ## Sensu Helpers
    def self.send_sensu_payload(name, output, status, ttl=86400, handler='pagerduty')
      sensu_hash = {
          name: name,
          output: output,
          handler: handler,
          ttl: ttl,
          status: status
      }

      UDPSocket.new.send(sensu_hash.to_json, 0, '127.0.0.1', 3030)
    end

    def send_sensu_payload(*args)
      CPE::Helpers.send_sensu_payload *args
    end

    # Arcana Secret Helpers
    def arcana_client
      @@client ||= Arcana::Client.new("litany/#{node.chef_environment}", get_region, node['arcana']['s3_bucket'], node.chef_environment)
    end

    def arcana_read(name)
      unless node['arcana'] && node['arcana']['s3_bucket']
        raise 'node["arcana"]["s3_bucket"] isn\'t defined!'
      end

      arcana_client.decrypt(name)
    end
  end
end

Chef::Recipe.send(:include, CPE::Helpers)
Chef::Provider.send(:include, CPE::Helpers)
Chef::Resource.send(:include, CPE::Helpers)
Chef::Mixin::Template::TemplateContext.send(:include, CPE::Helpers)
Chef::Node::Attribute.send(:include, CPE::Helpers)