# frozen_string_literal: true

property :instance_name, String, name_property: true

property :checksum, String
property :type, Symbol, equal_to: [:meta, :data], required: true
property :version, String

default_action :install

action :install do
  validate

  remote_file cached_file_path do
    source source_url
    checksum version_checksum
  end

  package new_resource.instance_name do
    options package_options unless package_options.nil?
    provider package_provider
    source cached_file_path
    action [:install, :upgrade]
  end
end

action :remove do
  validate strict: false

  remote_file cached_file_path do
    action :delete
  end

  package new_resource.instance_name do
    action platform_family?('debian') ? :purge : :remove
  end
end

action_class do
  def cached_file_path
    ::File.join(Chef::Config[:file_cache_path], file_name)
  end

  def file_base_name
    # Note: I'm not sure if the '#{version}-c${version}' pattern will hold true for future versions
    "influxdb-#{new_resource.type}_#{package_version}-c#{package_version}"
  end

  def file_name
    # Add platform_family logic to support more than debian
    "#{file_base_name}_amd64.deb"
  end

  def package_options
    # Add platform_family logic to support more than debian, returning nil is valid
    '--force-confdef --force-confold'
  end

  def package_provider
    # Add platform_family logic to support more than debian
    :dpkg_package
  end

  def package_version
    property_is_set?(:version) ? version : node['influxdb']['version']
  end

  def source_url
    "#{node['influxdb']['download_repository']}/#{file_name}"
  end

  def validate(strict: true)
    # Add platform_family logic to support more than debian
    raise "Platform family `#{node['platform_family']}` is unsupported." unless platform_family? 'debian'
    raise "You must provide either the `version` property or an attribute at `node[influxdb]['version']`." if package_version.nil?
    raise "You must provide either the `checksum` property or an attrbibute at `node['influxdb']['checksums']['#{new_resource.type}']['#{node['platform_family']}']['#{package_version}']`." if strict && version_checksum.nil?
  end

  def version_checksum
    property_is_set?(:checksum) ? checksum : node['influxdb']['checksums'].dig(new_resource.type, node['platform_family'], package_version)
  end
end
