#
# Cookbook:: sds_sensu
# Recipe:: default
#
# Copyright:: 2017, Curse, All Rights Reserved.

platform_family = node['platform_family']

execute 'systemctl daemon-reload' do
  action :nothing
end

case platform_family
  when "debian"
    include_recipe "apt"

    apt_repository "sensu" do
      uri node["sensu"]['apt_repo_url']
      distribution node["sensu"]["apt_repo_codename"] || node["lsb"]["codename"]
      components node["sensu"]["use_unstable_repo"] ? ["unstable"] : ["main"]
      key 'https://sensu.global.ssl.fastly.net/apt/pubkey.gpg'
      action :add
    end

    package_options = '--force-yes -o Dpkg::Options::="--force-confdef" -o Dpkg::Options::="--force-confnew"'

    package "sensu" do
      version node["sensu"]["version"]
      options package_options
      notifies :run, 'execute[systemctl daemon-reload]', :immediately
    end
  when 'rhel', 'fedora', 'amazon'
    repo = yum_repository "sensu" do
      description "sensu monitoring"
      repo = node["sensu"]["use_unstable_repo"] ? "yum-unstable" : "yum"
      baseurl "#{node['sensu']['yum_repo_url']}/#{repo}/$releasever/$basearch/"
      action :add
    end
    repo.gpgcheck(false) if repo.respond_to?(:gpgcheck)

    # As of 0.27 we need to suffix the version string with the platform major
    # version, e.g. ".el7". Override default via node["sensu"]["version_suffix"]
    # attribute.
    yum_package "sensu" do
      version lazy { Sensu::Helpers.redhat_version_string(
          node["sensu"]["version"],
          node["platform_version"],
          node["sensu"]["version_suffix"]
      )}
      allow_downgrade true
    end
  else
    raise "Unsupported Linux platform family #{platform_family}"
end

template '/etc/systemd/system/sensu-client.service' do
  source 'sensu-client.service.erb'
  owner 'root'
  group 'root'
  mode 00644
  helper(:sensu) { node['sensu'] }
  notifies :run, 'execute[systemctl daemon-reload]', :immediately
  notifies :restart, node['sensu']['service_cmd'], :delayed
end

directory '/etc/systemd/system/sensu-client.service.d' do
  recursive true
  action :delete
end

template '/etc/default/sensu' do
  source 'sensu.erb'

  owner 'root'
  group 'root'
  mode 00644

  helper(:sensu) { node['sensu'] }
end

%w(log_directory config_directory secret_directory).each do |dir|
  directory node['sensu'][dir] do
    owner node['sensu']['user']
    group node['sensu']['group']
    mode 00700

    recursive true
  end
end

file 'sensu/config.json' do
  path File.join(node['sensu']['config_directory'], 'config.json')
  content node['sensu']['common_config'] ? JSON.pretty_generate(node['sensu']['common_config']) : ''

  owner node['sensu']['user']
  group node['sensu']['group']
  mode 00640
end

%w(conf.d plugins).each do |folder|
  directory File.join(node['sensu']['config_directory'], folder) do
    owner node['sensu']['user'] unless node['platform'].eql? 'windows'
    group node['sensu']['group'] unless node['platform'].eql? 'windows'
    mode 00750
    recursive true

  end
end

file 'sensu/client.json'  do
  path File.join(node['sensu']['config_directory'], 'conf.d', 'client.json')
  content lazy { JSON.pretty_generate(:client => node['sensu']['client_config']) }

  owner node['sensu']['user'] unless node['platform'].eql? 'windows'
  group node['sensu']['group'] unless node['platform'].eql? 'windows'
  mode 00640

  notifies :restart, node['sensu']['service_cmd'], :delayed
end

sds_sensu_gem 'sensu-transport-snssqs-ng'

service 'sensu-client' do
  supports :restart => true, :status => true
  action [:enable, :start]

  retries 2
  retry_delay 2

  ignore_failure true

  subscribes :restart, 'file[sensu/config.json]', :delayed
end

directory File.join(node['sensu']['config_directory'], 'extensions')
