#
# Cookbook:: sds_vault
# Recipe:: default
#
# Copyright:: 2017, Curse, All Rights Reserved.

base_dir = node['vault']['base_dir']

directory base_dir do
  recursive true
end

user 'vault' do
  manage_home true
  home base_dir
  shell '/bin/bash'
  password ''
  notifies :run, 'execute[del-vault-pw]', :immediately
end

execute 'del-vault-pw' do
  command '/usr/bin/passwd -d vault'
  action :nothing
end

%W(
  packages
  ssl
  bin
  etc
  log
  data
).each do |dir|
  directory "#{base_dir}/#{dir}" do
    mode 0700
    owner 'vault'
    group 'vault'
    recursive true
  end
end

package 'unzip'

remote_file "#{base_dir}/packages/#{File.basename(node['vault']['package_url'])}" do
  source node['vault']['package_url']
  checksum node['vault']['package_sum']
  notifies :run, 'execute[extract vault]', :immediately
end

execute 'extract vault' do
  command "unzip #{base_dir}/packages/#{File.basename(node['vault']['package_url'])} && chown -R vault:vault #{base_dir}"
  cwd "#{base_dir}/bin"
  action :nothing
  notifies :restart, 'service[vault]', :delayed
end

template "#{base_dir}/bin/unseal.sh" do
  source 'unseal.sh.erb'
  helper(:base_dir) { base_dir }
  mode 0700
end

template "#{base_dir}/bin/step_down.sh" do
  source 'step_down.sh.erb'
  helper(:base_dir) { base_dir }
  mode 0700
end


template '/etc/init/vault-unseal.conf' do
  source 'unseal-upstart.conf.erb'
  mode 0600
  action :delete
end

template '/usr/lib/systemd/system/vault.service' do
  source 'vault.systemd.service.erb'
  helper(:base_dir) { base_dir }
end

template "#{base_dir}/etc/vault.hcl" do
  source 'vault.hcl.erb'
  helper(:bind_address) { node['ipaddress'] }
  helper(:consul_address) { node['ipaddress'] }
  helper(:cert_file) { node['vault']['cert_file'] }
  helper(:key_file) { node['vault']['key_file'] }
  notifies :restart, 'service[vault]', :delayed
end

service 'vault' do
  action [:enable, :start]
end

cron 'unseal vault' do
  user 'root'
  command %W{
          #{base_dir}/bin/unseal.sh > /dev/null
  }.join(' ')
end
