# frozen_string_literal: true

require 'clamp'
require 'aws-sdk-core'

require 'arcana'

module Arcana
  module CLI
    class Base < Clamp::Command

      option ['-n', '--material-name'], 'MATERIAL NAME', 'The name of the material (secret) you are targeting', required: true
      option ['-b', '--bucket-name'], 'BUCKET NAME', 'The name of the bucket that the secret is stored in.', required: true
      option ['-s', '--service-id'], 'SERVICE ID', 'The id of the service that this secret is related to.  In Litany this will look like <service name>-<stage name>-<project name>-<environment name>.', required: true
      option ['-p', '--system-prefix'], 'SYSTEM PREFIX', 'The system prefix for this service. In Litany these look like litany/<environment name>.', required: true

      option ['-P', '--profile'], 'AWS PROFILE', 'The name of the AWS profile to use from your ~/.aws/credentials file.', default: nil
      option ['-R', '--region'], 'AWS REGION', 'The name of the AWS region in which this secret resides.  Will be inferred for Litany services.', default: nil

      # Returns the Arcana client to interact with.
      # @return [Arcana::Client]
      private def arcana_client
        region_id = Arcana.region_id(region, service_id)

        if region_id.nil?
          puts 'Unable to determine a valid aws region id for this secret.'
          exit(1)
        end

        @arcana_client ||= Client.new(bucket: bucket_name, service_id: service_id, system_prefix: system_prefix, aws_profile: profile, aws_region: region_id)
      end
    end
  end
end