job {
    name 'mobile-web-device-tests'
    using 'TEMPLATE-autobuild'
    logRotator {
        daysToKeep(14)
    }
    publishers {
      archiveArtifacts('.manta/test-results.xml')
      archiveJunit('.manta/test-results.xml') {
        allowEmptyResults()
      }
      reportQuality('mobile-web/channel-lambda', '.manta', '*.xml')
    }
    scm {
        git {
            remote {
                github 'mobile-web/channel-lambda', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }
    steps {
        shell 'rm -rf .manta/'
        shell 'manta -v -proxy'
        saveDeployArtifact 'mobile-web/channel-lambda', '.manta'
    }
    throttleConcurrentBuilds {
      maxPerNode 0
    }
    wrappers {
      colorizeOutput(colorMap = 'xterm')
      environmentVariables {
        env('ENVIRONMENT', 'development')
      }
    }
}

job {
    name "TEMPLATE-mobile-web-deploy"
    using "TEMPLATE-deploy-aws"

    logRotator {
        daysToKeep(14)
    }

    steps {
        downloadDeployArtifact 'mobile-web/channel-lambda'
        shell """
            set +x
            export AWS_ACCESS_KEY_ID=\$MWC_AWS_ACCESS_KEY
            export AWS_SECRET_ACCESS_KEY=\$MWC_AWS_SECRET_KEY
            set -x
            sh scripts/upload -s "\$BUCKET" -a "\$ALIAS"
        """
    }
}

def stagingEnvironments = [
    public: 'STAGING',
    cap: 'CAP',
    cascade: 'CASCADE',
    lake: 'LAKE',
    sand: 'SAND',
    wooded: 'WOODED'
]

stagingEnvironments.each{ key, value ->
    // Clean-deploy will fire this job
    job {
        name "mobile-web-deploy-staging-${key}"
        using 'TEMPLATE-mobile-web-deploy'

        parameters {
            stringParam('ALIAS', value, 'version alias to update')
            stringParam('BUCKET', 'twitch-mobile-dev-static', 'bucket to hold static assets')
        }

        wrappers {
            credentialsBinding {
            file('AWS_CONFIG_FILE', 'aws_config')
            string 'MWC_AWS_ACCESS_KEY', 'mobile-web-dev-access-key'
            string 'MWC_AWS_SECRET_KEY', 'mobile-web-dev-secret-key'
            }
        }
    }
}

// Clean-deploy will fire this job
job {
    name 'mobile-web-deploy-production'
    using 'TEMPLATE-mobile-web-deploy'

    parameters {
        stringParam('ALIAS', 'PRODUCTION', 'version alias to update')
        stringParam('BUCKET', 'twitch-mobile-static', 'bucket to hold static assets')
    }

    wrappers {
        credentialsBinding {
          file('AWS_CONFIG_FILE', 'aws_config')
          string 'MWC_AWS_ACCESS_KEY', 'mobile-web-access-key'
          string 'MWC_AWS_SECRET_KEY', 'mobile-web-secret-key'
        }
    }
}

job {
    name "mobile-web-device-tests"
    using 'TEMPLATE-autobuild'
    wrappers {
        credentialsBinding {
            usernamePassword('SAUCE_USERNAME', 'SAUCE_KEY', 'mtf sauce-creds')
        }
    }
    scm {
        git {
            remote {
                github 'mobile-web/channel-lambda', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }

    steps {
        shell """
            |cd packages/mobile-web-tests
            |cat <<EOF>sauce_config.sh
            |export sauce_username=$SAUCE_USERNAME
            |export sauce_password=$SAUCE_KEY
            |export test_results_folder="\$(pwd)/output"
            |export build_number="$BUILD_NUMBER"
            |EOF
            
            |echo "$BUILD_NUMBER"
            
            |# Run the Build
            |set +x
            |set +e
            
            |cd ../..
            |manta -f packages/mobile-web-tests/build.json -v
            |python .manta/packages/mobile-web-tests/env/bin/sauce_connect.py -stop -u $SAUCE_USERNAME -k $SAUCE_KEY -tunnel $BUILD_NUMBER  
         """.stripMargin()
    }
    throttleConcurrentBuilds {
        maxPerNode 2
    }
    colorizeOutput(colorMap = 'xterm')
    timestamps()

    publishers {
        archiveArtifacts('.manta/packages/mobile-web-tests/tests/output/**/*')
        archiveJunit('.manta/packages/mobile-web-tests/output.xml')
    }
}
