"""Implements an object capable of controlling the channel list page.

Classes:
    ChannelPage: A representation of the channel list page.
"""

from fwk.base.locators import ByClass
from fwk.base.locators import ByCss
from fwk.base.base_page import BasePage


class ChannelPage(BasePage):
    """A utility class that can manipulate the channel list page for tests."""

    #  Elements definition
    PLACE_HOLDER = ByClass('player-holder')
    CHANNEL_LOGO = ByClass('info-box--logo-container')
    CHANNEL_METADATA = ByClass('info-box--metadata')
    CHAT_PANE = ByClass('chat-pane')
    CHAT_STATUS = ByClass('chat-status')
    TOP_NAV = ByClass('mw-top-nav')
    NAV_ICON = ByCss('div.mw-top-nav__menu button.tw-button-icon')
    NAV_CONTENTS = ByCss('div.tw-balloon div.tw-c-text-link')
    TWITCH_LOGO = ByCss('svg.tw-svg__asset--logotwitch')
    BROWSE_CHANNEL = ByCss('a.discover-more__channels')
    BROWSE_GAMES = ByCss('a.discover-more__games')
    SWITCH_TO_DESKTOP = ByClass('switch-to-desktop')
    PLAYER = ByClass('dynamic-player')

    def is_place_holder_present(self):
        """Return a bool describing if the player place holder is displayed."""
        return self.driver.find_element(*self.PLACE_HOLDER).is_displayed()

    def is_logo_present(self):
        """Return a bool if the channel logo is displayed on the page."""
        return self.driver.find_element(*self.CHANNEL_LOGO).is_displayed()

    def is_channel_metadata_present(self):
        """Return a bool if the channel metadata on the page is displayed."""
        return self.driver.find_element(*self.CHANNEL_METADATA).is_displayed()

    def channel_metadata_text(self):
        """Return the text of the metadata information on the player page."""
        self.wait_for_visible(self.CHANNEL_METADATA)
        return self.driver.find_element(*self.CHANNEL_METADATA).text

    def is_chat_pane_present(self):
        """Return a bool if the chat pane is displayed on the page."""
        return self.driver.find_element(*self.CHAT_PANE).is_displayed()

    def chat_status_text(self):
        """Return the status text in the chat area."""
        return self.driver.find_element(*self.CHAT_STATUS).text

    def is_player_icon_displayed(self):
        """Return a bool if the play button on the player screen is present."""
        return self.wait_for_visible(self.PLAYER_BUTTON)

    def is_top_nav_displayed(self):
        """Return a bool if the top nav of the page is displayed."""
        return self.driver.find_element(*self.TOP_NAV).is_displayed()

    def click_nav_icon(self):
        """Click on the nav icon containing more options."""
        self.driver.find_element(*self.NAV_ICON).click()

    def nav_menu_contents(self):
        """Return a list of the option test under the nav menu."""
        menu_contents = self.driver.find_elements(*self.NAV_CONTENTS)
        return [content.text for content in menu_contents]

    def is_twitch_logo_present(self):
        """Return bool if twitch logo is displayed on the page."""
        self.wait_for_visible(self.TWITCH_LOGO)
        return self.driver.find_element(*self.TWITCH_LOGO).is_displayed()

    def is_browse_all_channel_btn_present(self):
        """Return a bool if browse all channel button is displayed."""
        return self.driver.find_element(*self.BROWSE_CHANNEL).is_displayed()

    def browse_all_channel_btn_label(self):
        """Return a string of browse all channel button label."""
        return self.driver.find_element(*self.BROWSE_CHANNEL).text

    def is_browse_all_gmaes_btn_present(self):
        """Return a bool if browse all games button is displayed."""
        return self.driver.find_element(*self.BROWSE_GAMES).is_displayed()

    def browse_all_games_btn_label(self):
        """Return a string of browse all games button label."""
        return self.driver.find_element(*self.BROWSE_GAMES).text

    def switch_to_desktop_btn_label(self):
        """Return a string of switch to desktop button label."""
        return self.driver.find_element(*self.SWITCH_TO_DESKTOP).text

    def is_switch_to_desktop_btn_present(self):
        """Return a bool if switch to desktop button is displayed."""
        return self.driver.find_element(*self.SWITCH_TO_DESKTOP).is_displayed()

    def is_online_channel_loaded(self, channel_name):
        """Return a boolean if the channel is loaded."""
        self.wait_for_visible(self.PLAYER, 5)
        channel_present = channel_name in self.channel_metadata_text()
        status_present = 'Playing' in self.channel_metadata_text()
        return channel_present and status_present
