"""Implements an object capable of controlling the game directory page.

Classes:
    DirectoryPage: A representation of the game directory page.
"""

from fwk.base.locators import ByClass
from fwk.base.locators import ByCss
from fwk.base.base_page import BasePage


class DirectoryPage(BasePage):
    """A utility class that can manipulate the directory page for tests."""

    #  Elements definition
    GAME_LIST = ByClass('game-list')
    BOX_ART = ByClass("tw-card")
    CHANNEL_CARDS = ByClass("channel-thumbnail-card")
    CHANNEL_LIST = ByClass('channel-list')
    NAV_BAR = ByClass('mw-top-nav')
    CARD_TITLE = ByClass("tw-c-text")
    GAME_NAME = ByClass('tw-c-text-alt-2')
    VIEWERS_COUNT = ByCss('div.channel-thumbnail-card__preview__viewers span')
    CHANNEL_LIST = ByClass('channel-list')
    CHANNEL_PAGE = ByClass('channel-view')
    GAME_TITLE = ByCss('div.game-list h3.card__title')

    def is_game_list_present(self):
        """Check if the game list is displayed."""
        return self.driver.find_element(*self.GAME_LIST).is_displayed()

    def is_channel_list_present(self):
        """Check if channel list is displayed."""
        return self.driver.find_element(*self.CHANNEL_LIST).is_displayed()

    def click_first_game(self):
        """Click the first game from the game list."""
        self.driver.find_element(*self.BOX_ART).click()

    def is_nav_bar_displayed(self):
        """Return bool if nav bar displayed."""
        return self.driver.find_element(*self.NAV_BAR).is_displayed()

    def is_box_art_displayed(self):
        """Return bool if box art displayed."""
        return self.driver.find_element(*self.BOX_ART).is_displayed()

    def is_card_title_displayed(self):
        """Return bool if game title displayed."""
        return self.driver.find_element(*self.CARD_TITLE).is_displayed()

    def rotate_screen(self, orientation):
        """orientation it can be LANDSCAPE or PORTRAIT."""
        self.driver.orientation = orientation

    def cards_per_row(self):
        """Return int: Number of cards in a row."""
        list_width = self.driver.find_element(*self.GAME_LIST).size['width']
        card_width = self.driver.find_element(*self.BOX_ART).size['width']
        return list_width / card_width

    def click_on_channel(self, index=0):
        """Click on channel using index from channel directory page."""
        self.driver.find_elements(*self.CHANNEL_CARDS)[index].click()

    def game_name(self, index=0):
        """Return String Channel game name."""
        return self.driver.find_elements(*self.CHANNEL_CARDS)[index].text

    def is_viewers_count_displayed(self):
        """Return bool if viewers count displayed."""
        return self.driver.find_element(*self.VIEWERS_COUNT).is_displayed()

    def viewers_count(self, index=0):
        """Return Viewers count."""
        count = self.driver.find_elements(*self.VIEWERS_COUNT)[index].text
        count = count.split()[0]
        count = count.replace(',', '')
        return int(count)

    def channels_per_row(self):
        """Return number of cards in a row."""
        list_width = self.driver.find_element(*self.CHANNEL_LIST).size['width']
        cards = self.driver.find_element(*self.CHANNEL_CARDS)
        card_width = cards.size['width']
        return list_width / card_width

    def all_channels_name(self):
        """Return list Contains channel names."""
        self.wait_for_visible(self.VIEWERS_COUNT)
        all_channels = self.driver.find_elements(*self.CARD_TITLE)
        all_channels_name = [channel.text for channel in all_channels]
        all_channels_name.sort()
        return all_channels_name

    def is_on_channel_page(self):
        """Return bool if current page is channel page."""
        return self.driver.find_element(*self.CHANNEL_PAGE).is_displayed()

    def is_game_title_displayed(self):
        """Return bool if current page has game titles."""
        return self.driver.find_element(*self.GAME_TITLE).is_displayed()

    def card_title(self, index=0):
        """Return Card tiltle."""
        return self.driver.find_elements(*self.VIEWERS_COUNT)[index].text
