"""Implements an object capable of controlling the profile page.

Classes:
    ProfilePage: A representation of the profile page.
"""

from fwk.base.locators import ByClass
from fwk.base.locators import ByCss
from fwk.base.base_page import BasePage


class ProfilePage(BasePage):
    """A utility class that can manipulate the Profile page for tests."""

    TOP_NAV = ByClass('mw-top-nav')
    PROFILE_RED_INDICATOR = ByClass('tw-channel-status-indicator')
    PROFILE_ONLINE = ByCss('div.tw-flex-grow-2 span')
    PROFILE_EXPAND_BUTTON = ByClass('tw-svg__asset--plus')
    PROFILE_STATS = ByCss('.mw-channel-info-card div.tw-stat')
    PROFILE_AVATAR = ByCss('.mw-channel-info-card__avatar__image')
    PROFILE_CLOSE_BUTTON = ByClass('tw-svg__asset--close')
    PROFILE_DESCRIPTION = ByClass('mw-channel-info-card__details')
    HEADER_TEXT = ByCss('.tw-pd-x-1 h5 span')
    VIDEO_LIST = ByClass('mw-channel-profile-video-list')
    VIDEO_CARD_CSS = ByCss("a.mw-channel-profile-video-card")
    VIDEO_CARD = "a.mw-channel-profile-video-card"
    THUMBNAIL_IMG = ".mw-channel-profile-video-card__image"
    TIME = ".mw-channel-profile-video-card__time"
    VIDEO_DETAILS = ".mw-channel-profile-video-card__details"
    SHOW_MORE_BUTTON = ".mw_channel_profile_video_list__show_more button"

    # Past broadcast elements
    THUMB_CARD = 'div.mw-featured-content-card'
    THUMB_TITLE = 'div.mw-featured-content-card__details h5.tw-ellipsis'
    THUMB_CHANNEL = 'div.mw-featured-content-card__details p.tw-ellipsis'
    HEADER = 'div.mw-featured-content-card h5 span'
    THUMB_VIEWS = 'div.mw-featured-content-card__views span.tw-stat__value'
    PLAY_BUTTON = 'svg.tw-svg__asset--play'
    BROADCAST_CONTENT = ByClass("div.mw-featured-content-card")

    def profile_indicator_status(self):
        """Return String profile indicator status."""
        return self.driver.find_element(*self.PROFILE_ONLINE).text

    def click_profile_expand_btn(self):
        """Click on profile expand button if present."""
        if self.wait_for_visible(self.PROFILE_EXPAND_BUTTON):
            self.driver.find_element(*self.PROFILE_EXPAND_BUTTON).click()

    def is_profile_followers_count_present(self):
        """Return bool if the profile followers count present."""
        return self.driver.find_elements(*self.PROFILE_STATS)[0].is_displayed()

    def is_profile_viewers_count_present(self):
        """Return bool if the profile viewers count present."""
        return self.driver.find_elements(*self.PROFILE_STATS)[1].is_displayed()

    def is_displayed(self, element):
        """Return true if the element is displayed."""
        return self.driver.find_element(*element).is_displayed()

    def header_text(self):
        """Return a list of strings for headers."""
        header = self.driver.find_elements(*self.HEADER_TEXT)
        return [content.text for content in header]

    def section_on_page(self, section_name):
        """Return a element of video list."""
        header = self.driver.find_elements(*self.HEADER_TEXT)
        for index, value in enumerate(header):
            if section_name in value.text:
                return self.driver.find_elements(*self.VIDEO_LIST)[index]
        return None

    def video_cards(self, section_name):
        """Return array of video cards."""
        section = self.section_on_page(section_name)
        return section.find_elements_by_css_selector(self.VIDEO_CARD)

    def entity_details(self, section_name):
        """Return bool if entity details exist on page."""
        cards = self.video_cards(section_name)
        result_list = []
        for card in cards:
            thum_img = card.find_element_by_css_selector(self.THUMBNAIL_IMG)
            time = card.find_element_by_css_selector(self.TIME)
            vid_details = card.find_element_by_css_selector(self.VIDEO_DETAILS)
            result_list.append(thum_img.is_displayed())
            result_list.append(time.is_displayed())
            result_list.append(vid_details.is_displayed())
        return (False not in result_list) and ('' not in result_list)

    def click_show_more_button(self, section_name):
        """Click on show more button."""
        section = self.section_on_page(section_name)
        section.find_element_by_css_selector(self.SHOW_MORE_BUTTON).click()

    def food_page_profile_content(self):
        """Return bool if food page has valid content."""
        result_list = []
        thumb_card = self.driver.find_element_by_css_selector(self.THUMB_CARD)
        thmb_title = self.driver.find_element_by_css_selector(self.THUMB_TITLE)
        thumb_ch = self.driver.find_element_by_css_selector(self.THUMB_CHANNEL)
        thumb_view = self.driver.find_element_by_css_selector(self.THUMB_VIEWS)
        result_list.append(thumb_card.is_displayed())
        result_list.append(thmb_title.is_displayed())
        result_list.append(thumb_ch.is_displayed())
        result_list.append(thumb_view.is_displayed())
        return (False not in result_list) and ('' not in result_list)

    def channel_header_text(self):
        """Return string if has header."""
        self.wait_for_visible(self.HEADER_TEXT)
        return self.driver.find_element_by_css_selector(self.HEADER).text

    def broadcast_present(self):
        """Verify broadcast exists."""
        return self.wait_for_visible(self.BROADCAST_CONTENT)

    def play_button(self):
        """Return bool if player button present."""
        play_btn = self.driver.find_element_by_css_selector(self.PLAY_BUTTON)
        return play_btn.is_displayed()

    def is_profile_page(self, channel):
        """Return true if the current page is a profile page."""
        self.wait_for_visible(self.PROFILE_STATS)
        profile_url = channel + '/profile'
        return profile_url in self.driver.current_url

    def is_top_nav_displayed(self):
        """Return true if the top nav of the page is displayed."""
        return self.driver.find_element(*self.TOP_NAV).is_displayed()

    def is_past_broadcast_card_displayed(self):
        """Return a bool if the past broadcast section is displayed."""
        return self.wait_for_visible(self.BROADCAST_CONTENT)
