"""Implements an object capable of controlling the upsell page."""
import os
from fwk.base.locators import ByClass
from fwk.base.locators import ByCss
from fwk.base.base_page import BasePage

ANDROID_APP_DEEPLINK = """intent://m.twitch.tv/upsell/me?referrer\
=http%3A%2F%2Flocalhost.m.twitch.tv%3A3003%2Fupsell%2Fme#Intent;package=tv.twitch.android.app;scheme=twitch;\
S.market_referrer=utm_campaign%3Dmobile_upsell%26utm_source%3Dmwc;S.browser_fallback_url\
=market%3A%2F%2Fdetails%3Fid%3Dtv.twitch.android.app;end;"""
IOS_APP_DEEPLINK = """https://u.m.twitch.tv/upsell/me?c=mobile_upsell\
-primary_button"""


class UpsellPage(BasePage):
    """UpsellPage."""

    def __init__(self, driver):
        """Driver assignment to object."""
        self.driver = driver

    #  Elements definition
    UPSELL_PAGE_DIV = ByClass('upsell-page')
    UPSELL_PITCH = ByClass('upsell__pitch')
    TWITCH_LOGO = ByClass('twitch-logo')
    DESKTOP_MODE = ByClass('qa-decline_upsell')
    OPEN_IN_APP = ByCss('div.open-in-app a.tw-button')
    ERROR_MESSAGE = ByClass('core-error__message-container')

    def is_upsell_page(self):
        """Return a bool if the page has upsell banner."""
        return self.driver.find_element(*self.UPSELL_PAGE_DIV).is_displayed()

    def upsell_pitch_text(self):
        """Return a String of the text of the upsell pitch."""
        return self.driver.find_element(*self.UPSELL_PITCH).text

    def is_twitch_logo_present(self):
        """Return a bool if the page has twitch logo."""
        return self.driver.find_element(*self.TWITCH_LOGO).is_displayed()

    def is_switch_to_desktop_mode_link_displayed(self):
        """Return a bool if desktop mode button present."""
        desktop_text = self.driver.find_element(*self.DESKTOP_MODE).text
        return 'Switch to Desktop Mode' in desktop_text

    def is_open_in_app_present(self):
        """Return a bool if open in app button present."""
        open_in_app_text = self.driver.find_element(*self.OPEN_IN_APP).text
        return 'Open in App' in open_in_app_text

    def is_open_in_app_button_link_correct(self):
        """Return a bool if the link of the open in app button is correct."""
        open_in_app_element = self.driver.find_element(*self.OPEN_IN_APP)
        open_in_app_href = open_in_app_element.get_attribute('href')
        result = False
        if os.environ["platform"].lower() == 'android':
            result = ANDROID_APP_DEEPLINK == open_in_app_href
        elif os.environ["platform"].lower() == 'ios':
            result = IOS_APP_DEEPLINK == open_in_app_href
        return result

    def switch_to_desktop(self):
        """Switch to desktop site from mobile upsell page."""
        self.driver.execute_script("mobile: scroll", {"direction": "down"})
        element = self.driver.find_element(*self.DESKTOP_MODE)
        self.driver.execute_script("arguments[0].scrollIntoView(false);",
                                   element)
        element.click()

    def url_after_switch_to_desktop(self):
        """Return a String of the current URL after switching to desktop."""
        self.wait_for_visible(self.ERROR_MESSAGE, 20)
        return self.driver.current_url
