'use strict';

const path = require('path');
const process = require('process');
const webpack = require('webpack');
const WebpackShellPlugin = require('webpack-shell-plugin');
const ExtractTextPlugin = require('extract-text-webpack-plugin');

const gitHash = process.env.GIT_COMMIT || 'test_environment';

module.exports = {
  resolve: {
    extensions: ['.json', '.ts', '.tsx', '.js', '.gql'],
    mainFields: ['webpack', 'browser', 'style', 'main'],
    modules: [path.resolve(process.cwd(), 'src'), 'node_modules'],
    alias: {
      mweb: path.resolve(process.cwd(), 'src'),
      mtest: path.resolve(process.cwd(), 'test'),
      mconfig: path.resolve(process.cwd(), 'config'),
    },
  },
  module: {
    rules: [
      {
        test: /\.tsx?$/,
        use: ['react-hot-loader/webpack', 'awesome-typescript-loader'],
        exclude: ['node_modules', 'dist'],
      },
      {
        test: /\.css$/,
        use: ExtractTextPlugin.extract({
          use: [
            {
              loader: 'css-loader',
              options: { importLoaders: 1 },
            },
            {
              loader: 'postcss-loader',
              options: {
                plugins: () => [require('autoprefixer')()],
              },
            },
          ],
          fallback: {
            loader: 'style-loader',
            options: {
              hmr: false,
            },
          },
        }),
      },
      {
        test: /\.s(?:a|c)ss$/,
        use: ExtractTextPlugin.extract({
          use: [
            {
              loader: 'css-loader',
              options: { importLoaders: 2 },
            },
            {
              loader: 'postcss-loader',
              options: {
                plugins: () => [require('autoprefixer')()],
              },
            },
            {
              loader: 'sass-loader',
              options: {
                includePaths: ['node_modules/twitch-core-ui/src/sass'],
              },
            },
          ],
          fallback: {
            loader: 'style-loader',
            options: {
              hmr: false,
            },
          },
        }),
      },
      {
        test: /\.js$/,
        use: {
          loader: 'babel-loader',
          options: {
            babelrc: false,
            presets: [require('babel-preset-es2015')],
            cacheDirectory: true,
          },
        },
        include: [
          path.resolve(process.cwd(), '../../node_modules/twitch-tmi.js/'),
        ],
      },
      {
        test: /\.gql$/,
        use: {
          loader: 'raw-loader',
        },
      },
    ],
  },
  plugins: [
    // This fixes an issue where es6-promise incorrectly adds this
    // dep and webpack tries to pick it up.
    new webpack.IgnorePlugin(/vertx/),
    // The encodings library tries to outsmart webpack and browserfy
    // by assigning the iconv package name to a string and then req'ing
    // that string. This outsmarts browserfy, but just makes webpack more
    // angry. The library doesn't actually even require iconv, so we can
    // safely ignore it.
    new webpack.ContextReplacementPlugin(/iconv_package/, 'll'),
    new ExtractTextPlugin({
      filename: 'style-[name].[contenthash].css',
      allChunks: true,
    }),
    new webpack.DefinePlugin({
      'process.env': {
        GIT_HASH: JSON.stringify(gitHash),
        REACT_SPINKIT_NO_STYLES: JSON.stringify(true),
      },
    }),
    new WebpackShellPlugin({
      onBuildEnd: ['yarn i18n:manage'],
    }),
  ],
};
