import * as React from 'react';

import {
  Layout,
  Card,
  Background,
  CardBody,
  AspectRatio,
  Text,
  TextType,
  Color,
} from 'twitch-core-ui';
import { CoreImageSize } from 'twitch-core-ui/src/components/core-image/component';
import { FormattedMessage, injectIntl, InjectedIntl } from 'react-intl';

import { TrackedLink } from 'mweb/common/tracking/trackedLink';
import { buildChannelDirectoryPathFromDecodedGame } from 'mweb/common/utils/pathBuilders';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { getBoxArtSrcAndSrcSet } from 'mweb/common/utils/imageUtils';
import { TrackingInteractionContentProps } from 'mweb/common/tracking/withTracking';
import { asListItem } from 'mweb/common/components/asListItem';

export interface BoxArtCardOwnProps extends TrackingInteractionContentProps {
  title: string;
  imageTemplate: string;
  isLatencyCritical: boolean;
  viewCount: number;
  srcSetSizes: CoreImageSize[];
}

export interface BoxArtCardProps extends BoxArtCardOwnProps {
  intl: InjectedIntl;
}

export class BoxArtCardBase extends React.PureComponent<BoxArtCardProps> {
  render(): JSX.Element {
    return (
      <TrackedLink
        to={buildChannelDirectoryPathFromDecodedGame(this.props.title)}
        interactionContent={this.props.interactionContent}
        className="box-art-card"
      >
        <Card background={Background.Base} elevation={1}>
          <CardImageWrapper
            {...getBoxArtSrcAndSrcSet(this.props.imageTemplate)}
            latencyTracked={this.props.isLatencyCritical}
            alt={this.props.title}
            aspect={AspectRatio.BoxArt}
            key={this.props.imageTemplate}
            sizes={this.props.srcSetSizes}
          />
          <CardBody>
            <Layout margin={{ x: 1, y: 0.5 }}>
              <Text
                type={TextType.H5}
                ellipsis={true}
                bold={true}
                color={Color.Base}
              >
                {this.props.title}
              </Text>
              <Text type={TextType.P} ellipsis={true} color={Color.Alt}>
                <FormattedMessage
                  id="x-viewers"
                  defaultMessage="{viewCount} viewers"
                  values={{
                    viewCount: this.props.intl.formatNumber(
                      this.props.viewCount,
                    ),
                  }}
                />
              </Text>
            </Layout>
          </CardBody>
        </Card>
      </TrackedLink>
    );
  }
}

export const BoxArtCard = injectIntl(BoxArtCardBase) as React.ComponentClass<
  BoxArtCardOwnProps
>;
export const BoxArtListItemCard = asListItem(BoxArtCard);
