import * as React from 'react';
import { buildChannelPath } from 'mweb/common/utils/pathBuilders';

import {
  Layout,
  Card,
  Background,
  CardBody,
  AspectRatio,
  Text,
  TextType,
  Color,
  Avatar,
  Display,
} from 'twitch-core-ui';
import { CoreImageSize } from 'twitch-core-ui/src/components/core-image/component';
import { FormattedMessage, injectIntl, InjectedIntl } from 'react-intl';

import { TrackedLink } from 'mweb/common/tracking/trackedLink';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { getChannelThumbnailSrcAndSrcSet } from 'mweb/common/utils/imageUtils';
import { TrackingInteractionContentProps } from 'mweb/common/tracking/withTracking';
import { asListItem } from 'mweb/common/components/asListItem';

if (process.env.BROWSER) {
  require('./channelThumbnailCard.sass');
}

const UNKNOWN_THUMBNAIL_TEMPLATE =
  'http://static-cdn.jtvnw.net/previews-ttv/live_user_test_channel-{width}x{height}.jpg';

export interface ChannelThumbnailCardOwnProps
  extends TrackingInteractionContentProps {
  name: string;
  displayName: string;
  title: string | undefined;
  gameName: string | undefined;
  viewCount: number;
  isLatencyCritical: boolean;
  logoURL: string | undefined;
  srcSetSizes: CoreImageSize[];
  imageTemplate?: string;
}

export interface ChannelThumbnailCardProps
  extends ChannelThumbnailCardOwnProps {
  intl: InjectedIntl;
}

export class ChannelThumbnailCardBase extends React.PureComponent<
  ChannelThumbnailCardProps
> {
  render(): JSX.Element {
    return (
      <TrackedLink
        to={buildChannelPath(this.props.name)}
        interactionContent={this.props.interactionContent}
        className="channel-thumbnail-card"
      >
        <Card background={Background.Base} elevation={1}>
          <Layout className="channel-thumbnail-card__preview">
            <CardImageWrapper
              {...getChannelThumbnailSrcAndSrcSet(
                this.props.imageTemplate || UNKNOWN_THUMBNAIL_TEMPLATE,
              )}
              latencyTracked={this.props.isLatencyCritical}
              alt={this.props.gameName || `thumbnail for ${this.props.name}`}
              aspect={AspectRatio.Aspect16x9}
              key={this.props.imageTemplate}
              sizes={this.props.srcSetSizes}
            >
              <Layout className="channel-thumbnail-card__preview__viewers">
                <FormattedMessage
                  id="x-viewers"
                  defaultMessage="{viewCount} viewers"
                  values={{
                    viewCount: this.props.intl.formatNumber(
                      this.props.viewCount,
                    ),
                  }}
                />
              </Layout>
            </CardImageWrapper>
          </Layout>
          <CardBody>
            <Layout
              margin={{ x: 1 }}
              display={Display.Flex}
              className="channel-thumbnail-card__info"
            >
              <Layout flexShrink={0} margin={{ top: 1 }}>
                <Avatar
                  alt={`avatar for ${this.props.name}`}
                  size={36}
                  src={this.props.logoURL}
                />
              </Layout>
              <Layout
                margin={{ left: 1, y: 0.5 }}
                ellipsis={true}
                flexShrink={2}
              >
                <Text
                  type={TextType.H5}
                  bold={true}
                  color={Color.Base}
                  ellipsis={true}
                >
                  {this.props.title}
                </Text>
                <Text type={TextType.P} color={Color.Alt} ellipsis={true}>
                  {this.props.displayName}
                </Text>
                <Text type={TextType.P} color={Color.Alt2} ellipsis={true}>
                  {this.props.gameName}
                </Text>
              </Layout>
            </Layout>
          </CardBody>
        </Card>
      </TrackedLink>
    );
  }
}

export const ChannelThumbnailCard = injectIntl(
  ChannelThumbnailCardBase,
) as React.ComponentClass<ChannelThumbnailCardOwnProps>;
export const ChannelThumbnailListItemCard = asListItem(ChannelThumbnailCard);
