import * as React from 'react';
import { FormattedMessage } from 'react-intl';

import {
  buildGameDirectoryPath,
  buildChannelDirectoryPathFromDecodedGame,
} from 'mweb/common/utils/pathBuilders';
import { ALL_CHANNELS } from 'mweb/common/reducers/data/channels';
import { withTracking, TrackingProps } from 'mweb/common/tracking/withTracking';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

const INTERACTION_MEDIUM = 'discover_more';
export const INTERACTION_CONTENT_CHANNELS = 'channels';
export const INTERACTION_CONTENT_GAMES = 'games';

interface DiscoverMoreOwnProps {
  game: string | undefined;
}

export interface DiscoverMoreProps
  extends DiscoverMoreOwnProps,
    TrackingProps {}

export class DiscoverMoreBase extends React.PureComponent<
  DiscoverMoreProps,
  {}
> {
  get channelDirectoryLinkURL(): string {
    return buildChannelDirectoryPathFromDecodedGame(
      this.props.game || ALL_CHANNELS,
    );
  }

  get channelDirectoryText(): JSX.Element {
    if (!this.props.game) {
      return (
        <FormattedMessage
          id="discover-more--discover-all"
          defaultMessage="Browse all other channels!"
        />
      );
    }

    return (
      <FormattedMessage
        id="discover-more--discover-more-of-game"
        defaultMessage="Browse all other {game} channels!"
        values={{ game: this.props.game }}
      />
    );
  }

  render(): JSX.Element {
    return (
      <div className="discover-more">
        <TrackedLink
          className="discover-more__button discover-more__channels"
          to={this.channelDirectoryLinkURL}
          interactionContent={INTERACTION_CONTENT_CHANNELS}
        >
          {this.channelDirectoryText}
        </TrackedLink>
        <TrackedLink
          className="discover-more__button discover-more__games test-browse-all-games"
          to={buildGameDirectoryPath()}
          interactionContent={INTERACTION_CONTENT_GAMES}
        >
          <FormattedMessage
            id="discover-more--discover-more-games"
            defaultMessage="Browse all other games!"
          />
        </TrackedLink>
      </div>
    );
  }
}

export const DiscoverMore = extendInteractionMedium(
  withTracking(DiscoverMoreBase),
  INTERACTION_MEDIUM,
);
