import * as React from 'react';
import { StyledLayout, Button, Text, Background } from 'twitch-core-ui';
import {
  defineMessages,
  injectIntl,
  InjectedIntl,
  FormattedMessage,
} from 'react-intl';

import {
  ShareButton,
  ShareButtonDispatchProps,
} from 'mweb/common/components/shareButton';
import {
  RemindMeButton,
  RemindMeButtonValueProps,
  RemindMeButtonDispatchProps,
} from 'mweb/common/components/remindMeButton';
import { PremiereStatus } from 'mweb/common/reducers/data/events';
import { buildChannelPath } from 'mweb/common/utils/pathBuilders';

import { Layout, Display, FlexDirection } from 'twitch-core-ui';
import { TrackingProps, withTracking } from 'mweb/common/tracking/withTracking';
import { OpenInApp } from 'mweb/common/containers/openInApp';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';

if (process.env.BROWSER) {
  require('./eventActions.sass');
}

export const INTERACTION_CONTENT_WATCH_NOW = 'watch_now';
export const EVENT_ACTIONS_MEDIUM = 'event_actions';
export const CALL_TO_ACTION_IDENTIFIER = EVENT_ACTIONS_MEDIUM;
const INTERACTION_MEDIUM = 'actions';

const messages = defineMessages({
  shareEvent: {
    id: 'share-event',
    defaultMessage: 'Share Event',
  },
});

export interface EventActionsValueProps extends RemindMeButtonValueProps {
  channelName: string;
  premiereStatus?: PremiereStatus;
}

export interface EventActionsDispatchProps
  extends RemindMeButtonDispatchProps,
    ShareButtonDispatchProps {}

export interface EventActionsOwnProps
  extends EventActionsValueProps,
    EventActionsDispatchProps {}

export interface EventActionsProps extends EventActionsOwnProps, TrackingProps {
  intl: InjectedIntl;
}

export class EventActionsBase extends React.PureComponent<
  EventActionsProps,
  {}
> {
  get remindMeButton(): JSX.Element {
    return (
      <RemindMeButton
        startTime={this.props.startTime}
        endTime={this.props.endTime}
        title={this.props.title}
        description={this.props.description}
        isMobileOS={this.props.isMobileOS}
        appLocation={this.props.appLocation}
        handleNotificationRequest={this.props.handleNotificationRequest}
      />
    );
  }

  get openInAppPath(): string {
    return `/${this.props.channelName}`;
  }

  get watchNowButton(): JSX.Element {
    return (
      <StyledLayout className="event-watch-now" background={Background.Accent}>
        <Button
          linkTo={buildChannelPath(this.props.channelName)}
          onClick={this.props.trackClick({
            interactionContent: INTERACTION_CONTENT_WATCH_NOW,
          })}
        >
          <Text>
            <FormattedMessage
              id="event--watch-now"
              defaultMessage="Watch Now"
            />
          </Text>
        </Button>
      </StyledLayout>
    );
  }

  get pastEventButton(): JSX.Element {
    return (
      <Button disabled>
        <Text>
          <FormattedMessage
            id="event--past-event"
            defaultMessage="Past Event"
          />
        </Text>
      </Button>
    );
  }

  get primaryButton(): JSX.Element {
    if (
      !this.props.premiereStatus ||
      this.props.premiereStatus === PremiereStatus.Scheduled
    ) {
      return this.remindMeButton;
    } else if (this.props.premiereStatus === PremiereStatus.Started) {
      return this.watchNowButton;
    } else {
      return this.pastEventButton;
    }
  }

  get openInAppButton(): JSX.Element | false {
    return (
      this.props.isMobileOS &&
      this.props.premiereStatus === PremiereStatus.Started && (
        <StyledLayout background={Background.Accent}>
          <OpenInApp
            path={this.openInAppPath}
            callToActionIdentifier={CALL_TO_ACTION_IDENTIFIER}
            medium={EVENT_ACTIONS_MEDIUM}
          />
        </StyledLayout>
      )
    );
  }

  render(): JSX.Element {
    return (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Row}
        className="event-actions-container"
      >
        {this.primaryButton}
        {this.openInAppButton}
        <ShareButton
          shareSheetHeader={this.props.intl.formatMessage(messages.shareEvent)}
          title={this.props.title}
          appLocation={this.props.appLocation}
          handleShare={this.props.handleShare}
        />
      </Layout>
    );
  }
}

export const EventActions = injectIntl(
  extendInteractionMedium(withTracking(EventActionsBase), INTERACTION_MEDIUM),
) as React.ComponentClass<EventActionsOwnProps>;
