import * as React from 'react';
import { buildStringFromTemplate } from 'mweb/common/utils/imageUtils';
import { FormattedMessage } from 'react-intl';
import {
  Layout,
  StyledLayout,
  InjectLayout,
  FlexDirection,
  Background,
  Avatar,
  Text,
  TextType,
  TextTransform,
  Color,
  AlignItems,
  Display,
  Interactable,
} from 'twitch-core-ui';

import {
  buildChannelDirectoryPathFromDecodedGame,
  buildChannelPath,
  buildVODPath,
} from 'mweb/common/utils/pathBuilders';

import { BaseVideoDetails } from 'mweb/common/reducers/data/baseVideoDetails';
import { GameDetails } from 'mweb/common/reducers/data/games';
import VideoCard from 'mweb/common/components/videoCard';
import { TrackingProps, withTracking } from 'mweb/common/tracking/withTracking';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';

if (process.env.BROWSER) {
  require('./eventPanels.sass');
}

const BOX_ART_HEIGHT = 58;
const BOX_ART_WIDTH = 42;
const INTERACTION_CONTENT_VOD = 'vod';
export const INTERACTION_CONTENT_GAME = 'game';
export const INTERACTION_CONTENT_CHANNEL = 'channel';
const INTERACTION_MEDIUM = 'panels';

export interface EventPanelsOwnProps {
  logoURL: string | undefined;
  channelDisplayName: string;
  game: GameDetails | undefined;
  channelName: string;
  video: BaseVideoDetails | undefined;
}

export interface EventPanelsProps extends EventPanelsOwnProps, TrackingProps {}

export class EventPanels extends React.Component<EventPanelsProps, {}> {
  get vodInfo(): JSX.Element | undefined {
    const video = this.props.video;
    return (
      video && (
        <StyledLayout
          elevation={1}
          margin={{ bottom: 1 }}
          className="event-vod-info"
        >
          <VideoCard
            key={video.id}
            title={video.title}
            imageURL={video.thumbnailURL}
            gameName={video.game}
            viewCount={video.viewCount}
            createdAt={video.date}
            formattedLength={video.formattedLength}
            borderTop={false}
            linkTo={buildVODPath(video.id)}
            interactionContent={INTERACTION_CONTENT_VOD}
          />
        </StyledLayout>
      )
    );
  }

  get channelInfo(): JSX.Element {
    return (
      <Interactable
        linkTo={buildChannelPath(this.props.channelName)}
        onClick={this.props.trackClick({
          interactionContent: INTERACTION_CONTENT_CHANNEL,
        })}
      >
        <StyledLayout
          className="event-channel-info"
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
          alignItems={AlignItems.Center}
          background={Background.Base}
          elevation={1}
          padding={1}
          margin={{ bottom: 1 }}
        >
          <Avatar
            alt={this.props.channelDisplayName}
            src={this.props.logoURL}
            size={40}
          />
          <InjectLayout padding={{ left: 1 }}>
            <Text type={TextType.Span} color={Color.Link}>
              {this.props.channelDisplayName}
            </Text>
          </InjectLayout>
        </StyledLayout>
      </Interactable>
    );
  }

  get gameInfo(): JSX.Element | undefined {
    const game = this.props.game;
    if (game) {
      const buildBoxArtURL = (multiplier: number) =>
        buildStringFromTemplate(
          game.boxArtURL,
          multiplier * BOX_ART_WIDTH,
          multiplier * BOX_ART_HEIGHT,
        );

      return (
        <Interactable
          linkTo={buildChannelDirectoryPathFromDecodedGame(game.name)}
          onClick={this.props.trackClick({
            interactionContent: INTERACTION_CONTENT_GAME,
          })}
        >
          <StyledLayout
            className="event-game-info"
            display={Display.Flex}
            flexDirection={FlexDirection.Row}
            alignItems={AlignItems.Center}
            background={Background.Base}
            elevation={1}
            padding={1}
          >
            <img
              alt={game.name}
              className="event-channel-boxart"
              src={buildBoxArtURL(1)}
              srcSet={`
          ${buildBoxArtURL(1)},
          ${buildBoxArtURL(2)} 2x,
          ${buildBoxArtURL(3)} 3x,
          ${buildBoxArtURL(4)} 4x
        `}
            />
            <InjectLayout padding={{ left: 1 }}>
              <Text type={TextType.Span} color={Color.Link}>
                {game.name}
              </Text>
            </InjectLayout>
          </StyledLayout>
        </Interactable>
      );
    }
  }

  render(): JSX.Element {
    return (
      <Layout className="event-info">
        {this.vodInfo}

        <InjectLayout margin={{ left: 1, bottom: 0.5 }}>
          <Text
            type={TextType.H5}
            color={Color.Alt2}
            transform={TextTransform.Uppercase}
          >
            <FormattedMessage id="event--starring" defaultMessage="Starring" />
          </Text>
        </InjectLayout>
        {this.channelInfo}

        <InjectLayout margin={{ left: 1, bottom: 0.5 }}>
          <Text
            type={TextType.H5}
            color={Color.Alt2}
            transform={TextTransform.Uppercase}
          >
            <FormattedMessage id="event--game" defaultMessage="Game" />
          </Text>
        </InjectLayout>
        {this.gameInfo}
      </Layout>
    );
  }
}

export default extendInteractionMedium(
  withTracking(EventPanels),
  INTERACTION_MEDIUM,
);
