import * as React from 'react';
import { FormattedMessage } from 'react-intl';

import {
  buildChannelPath,
  buildChannelDirectoryPathFromDecodedGame,
  buildChannelProfilePath,
} from 'mweb/common/utils/pathBuilders';
import { withTracking, TrackingProps } from 'mweb/common/tracking/withTracking';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

if (process.env.BROWSER) {
  require('./infoBox.sass');
}

const INTERACTION_MEDIUM = 'info_box';
export const INTERACTION_CONTENT_LOGO = 'logo';
export const INTERACTION_CONTENT_GAME_NAME = 'game';
export const INTERACTION_CONTENT_HOSTED_CHANNEL = 'hosted_Channel';

export interface InfoBoxProps extends TrackingProps {
  channelName: string | undefined;
  channelDisplayName: string | undefined;
  logoUrl: string | undefined;
  game?: string;
  hostedChannel?: string;
  hostedChannelDisplayName?: string;
  vodTitle?: string;
}

// TODO: make separate Channel and VOD info boxes
export class InfoBox extends React.PureComponent<InfoBoxProps, {}> {
  render(): JSX.Element {
    const logo =
      this.props.logoUrl && this.props.channelName ? (
        <div className="info-box--logo-container">
          <TrackedLink
            to={buildChannelProfilePath(this.props.channelName)}
            interactionContent={INTERACTION_CONTENT_LOGO}
          >
            <img src={this.props.logoUrl} alt={this.props.channelDisplayName} />
          </TrackedLink>
        </div>
      ) : null;
    return (
      <div className="info-box qa-info-box">
        {logo}
        <div className="info-box--metadata qa-info-box--metadata">
          {this.channelName()}
          {this.title()}
        </div>
      </div>
    );
  }

  channelName(): JSX.Element[] {
    const elements = [
      <h2 key="main-channel">{this.props.channelDisplayName}</h2>,
    ];

    if (this.props.hostedChannelDisplayName && this.props.hostedChannel) {
      elements.push(
        <h2 key="hosted-channel">
          <FormattedMessage
            id="info-box--hosted-channel-name"
            defaultMessage="hosting {hostedChannel}"
            values={{
              hostedChannel: (
                <TrackedLink
                  to={buildChannelPath(this.props.hostedChannel)}
                  interactionContent={INTERACTION_CONTENT_HOSTED_CHANNEL}
                >
                  {this.props.hostedChannelDisplayName}
                </TrackedLink>
              ),
            }}
          />
        </h2>,
      );
    }
    return elements;
  }

  title(): JSX.Element | undefined {
    if (this.props.vodTitle) {
      return <p className="info-box--playing-info">{this.props.vodTitle}</p>;
    }
    if (this.props.game) {
      return (
        <p className="info-box--playing-info">
          <FormattedMessage
            id="info-box--playing-info"
            defaultMessage="Playing {game}"
            values={{
              game: (
                <TrackedLink
                  to={buildChannelDirectoryPathFromDecodedGame(this.props.game)}
                  interactionContent={INTERACTION_CONTENT_GAME_NAME}
                >
                  {this.props.game}
                </TrackedLink>
              ),
            }}
          />
        </p>
      );
    }
    return undefined;
  }
}

export default extendInteractionMedium(
  withTracking(InfoBox),
  INTERACTION_MEDIUM,
);
