import * as React from 'react';
import { defineMessages, injectIntl, InjectedIntl } from 'react-intl';
import {
  SVGAsset,
  Interactable,
  ButtonIcon,
  ButtonSize,
  Layout,
  Text,
  TextType,
  Color,
  Position,
  FlexDirection,
  AlignItems,
  JustifyContent,
  ZIndex,
  Display,
  ButtonIconType,
  StyledLayout,
  InjectLayout,
  Background,
} from 'twitch-core-ui';

import { Twitter } from 'mweb/common/components/svg/twitter';
import { Facebook } from 'mweb/common/components/svg/facebook';
import { Reddit } from 'mweb/common/components/svg/reddit';
import { Copy } from 'mweb/common/components/svg/copy';
import { Location } from 'mweb/common/reducers/app';
import { ShareSheetItem } from 'mweb/common/components/shareSheetItem';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';

if (process.env.BROWSER) {
  require('./shareSheet.sass');
}

const messages = defineMessages({
  copy: {
    id: 'share-button--copy-url',
    defaultMessage: 'Copy',
  },
});

const INTERACTION_MEDIUM = 'sharing_dock';
export const TWITTER = 'twitter';
export const FACEBOOK = 'facebook';
export const REDDIT = 'reddit';
export const URL = 'url';
const ICON_SIZE = 32;

export interface ShareSheetOwnProps {
  header: string;
  isVisible: boolean;
  title: string;
  appLocation: Location;
  toggleSharingDock: () => void;
  handleShare: (medium: string, location: Location) => void;
}

export interface ShareSheetProps extends ShareSheetOwnProps {
  intl: InjectedIntl;
}

export class ShareSheetBase extends React.PureComponent<ShareSheetProps, {}> {
  handleShare(medium: string, location: Location): () => void {
    return () => {
      this.props.toggleSharingDock();
      this.props.handleShare(medium, location);
    };
  }

  get twitterURL(): string {
    const message = `${this.props.title} ${document.location.href}`;
    return `https://twitter.com/intent/tweet?text=${encodeURIComponent(
      message,
    )}`;
  }

  get redditURL(): string {
    return `http://www.reddit.com/submit?url=${encodeURIComponent(
      window.location.href,
    )}&title=${encodeURIComponent(this.props.title)}`;
  }

  get facebookURL(): string {
    return `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(
      window.location.href,
    )}`;
  }

  handleContentClick(e: React.MouseEvent<HTMLDivElement>): void {
    e.stopPropagation();
  }

  get shareHeader(): JSX.Element {
    return (
      <StyledLayout
        className="share-sheet__header"
        display={Display.Flex}
        flexDirection={FlexDirection.Row}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Between}
        padding={{ left: 1 }}
        borderBottom
      >
        <Text type={TextType.H5} color={Color.Alt2} bold>
          {this.props.header}
        </Text>
        <ButtonIcon
          ariaLabel="Close share sheet"
          size={ButtonSize.Large}
          onClick={this.props.toggleSharingDock}
          icon={SVGAsset.Close}
          type={ButtonIconType.Secondary}
        />
      </StyledLayout>
    );
  }

  get shareOptions(): JSX.Element {
    return (
      <InjectLayout
        display={Display.Flex}
        justifyContent={JustifyContent.Around}
        flexDirection={FlexDirection.Column}
        padding={{ y: 0.5 }}
        fullHeight
      >
        <ul className="share-sheet__list">
          <li>
            <ShareSheetItem
              interactionContent={TWITTER}
              displayName="Twitter"
              url={this.twitterURL}
              handleShare={this.handleShare(TWITTER, this.props.appLocation)}
            >
              <Twitter height={ICON_SIZE} width={ICON_SIZE} />
            </ShareSheetItem>
          </li>
          <li>
            <ShareSheetItem
              interactionContent={REDDIT}
              displayName="Reddit"
              url={this.redditURL}
              handleShare={this.handleShare(REDDIT, this.props.appLocation)}
            >
              <Reddit height={ICON_SIZE} width={ICON_SIZE} />
            </ShareSheetItem>
          </li>
          <li>
            <ShareSheetItem
              interactionContent={FACEBOOK}
              displayName="Facebook"
              url={this.facebookURL}
              handleShare={this.handleShare(FACEBOOK, this.props.appLocation)}
            >
              <Facebook height={ICON_SIZE} width={ICON_SIZE} />
            </ShareSheetItem>
          </li>
          <li>
            <ShareSheetItem
              interactionContent={URL}
              displayName={this.props.intl.formatMessage(messages.copy)}
              url={window.location.href}
              handleShare={this.handleShare(URL, this.props.appLocation)}
              isClipboard
            >
              <Copy height={ICON_SIZE} width={ICON_SIZE} />
            </ShareSheetItem>
          </li>
        </ul>
      </InjectLayout>
    );
  }

  render(): JSX.Element | false {
    return (
      this.props.isVisible && (
        <Layout
          className="share-sheet"
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          zIndex={ZIndex.Above}
          position={Position.Fixed}
          attachTop
          attachRight
          attachLeft
          attachBottom
        >
          <StyledLayout
            className="share-sheet__backdrop"
            display={Display.Flex}
            flexGrow={1}
            background={Background.AccentAlt2}
          >
            <Interactable onClick={this.props.toggleSharingDock} />
          </StyledLayout>
          <StyledLayout
            className="share-sheet__content"
            elevation={1}
            background={Background.Base}
          >
            {this.shareHeader}
            {this.shareOptions}
          </StyledLayout>
        </Layout>
      )
    );
  }
}

export const ShareSheet = injectIntl(
  extendInteractionMedium(ShareSheetBase, INTERACTION_MEDIUM),
) as React.ComponentClass<ShareSheetOwnProps>;
