import * as React from 'react';
import {
  FormattedDate,
  FormattedMessage,
  defineMessages,
  injectIntl,
  InjectedIntl,
} from 'react-intl';
import {
  AlignItems,
  Background,
  Display,
  FlexWrap,
  JustifyContent,
  Layout,
  Stat,
  StyledLayout,
  SVGAsset,
  Text,
  TextType,
  AspectRatio,
} from 'twitch-core-ui';

import {
  withTracking,
  TrackingInteractionContentProps,
  TrackingProps,
} from 'mweb/common/tracking/withTracking';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { getChannelThumbnailSrcAndSrcSet } from 'mweb/common/utils/imageUtils';

if (process.env.BROWSER) {
  require('./videoCard.sass');
}

const DEFAULT_ALT_TEXT = 'thumbnail for video';
const messages = defineMessages({
  videoLength: {
    id: 'video-length',
    defaultMessage: 'Video length',
  },
});

interface VideoCardOwnProps extends TrackingInteractionContentProps {
  title: string | undefined;
  imageURL: string;
  viewCount: number;
  formattedLength: string;
  createdAt: number;
  linkTo: string;
  gameName: string | undefined;
  borderTop: boolean;
}

export interface VideoCardProps extends VideoCardOwnProps, TrackingProps {
  intl: InjectedIntl;
}

export class VideoCard extends React.PureComponent<VideoCardProps, {}> {
  get linkContent(): JSX.Element {
    return (
      <StyledLayout
        background={Background.Base}
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Start}
        flexWrap={FlexWrap.NoWrap}
        padding={1}
        borderTop={this.props.borderTop}
        borderBottom
        fullWidth
      >
        <Layout className="mw-channel-profile-video-card__image" flexShrink={0}>
          <CardImageWrapper
            {...getChannelThumbnailSrcAndSrcSet(this.props.imageURL)}
            latencyTracked={false}
            alt={this.props.title || DEFAULT_ALT_TEXT}
            aspect={AspectRatio.Aspect16x9}
            key={this.props.imageURL}
            sizes={[{ size: '105px' }]}
          />
          <Layout className="mw-channel-profile-video-card__time">
            <Stat
              icon={SVGAsset.GlyphLength}
              label={this.props.intl.formatMessage(messages.videoLength)}
              value={this.props.formattedLength}
            />
          </Layout>
        </Layout>
        <Layout
          className="mw-channel-profile-video-card__details"
          margin={{ left: 1 }}
          ellipsis={true}
        >
          <Text type={TextType.H4} ellipsis={true}>
            {this.props.title}
          </Text>
          <Text ellipsis={true}>
            {this.props.gameName}
            &nbsp;&#124;&nbsp;
            <FormattedMessage
              id="views-count"
              defaultMessage="{viewCount, plural, one {1 view} other {{formattedViewCount} views}}"
              values={{
                viewCount: this.props.viewCount,
                formattedViewCount: this.props.intl.formatNumber(
                  this.props.viewCount || 0,
                  { maximumSignificantDigits: 3 },
                ),
              }}
            />
            &nbsp;&#124;&nbsp;
            <FormattedDate
              value={this.props.createdAt}
              year="numeric"
              month="short"
              day="numeric"
            />
          </Text>
        </Layout>
      </StyledLayout>
    );
  }

  outerClassName: string = 'mw-channel-profile-video-card';

  render(): JSX.Element {
    // short-term special case while clips are external
    if (this.props.linkTo.startsWith('https')) {
      return (
        <a
          href={this.props.linkTo}
          target="_blank"
          className={this.outerClassName}
          onClick={this.props.trackClick({
            interactionContent: this.props.interactionContent,
            interactionTargetPath: `/clips/${this.props.linkTo
              .split('/')
              .slice(-1)[0]}`,
          })}
        >
          {this.linkContent}
        </a>
      );
    }

    return (
      <TrackedLink
        className={this.outerClassName}
        to={this.props.linkTo}
        interactionContent={this.props.interactionContent}
      >
        {this.linkContent}
      </TrackedLink>
    );
  }
}

export default injectIntl(withTracking(VideoCard)) as React.ComponentClass<
  VideoCardOwnProps
>;

// TODO: when kill off default export, bring in asListItem verison (channelProfileVideoList)
