import * as React from 'react';
import { connect } from 'react-redux';
import { FormattedMessage } from 'react-intl';
import { Button, Layout } from 'twitch-core-ui';

import { buildDeepLinkURL } from 'mweb/common/utils/deepLink';
import { OS } from 'mweb/common/reducers/device';
import { Location } from 'mweb/common/reducers/app';
import { withLatencyTracker } from 'mweb/common/latency/withLatencyTracker';
import { withTracking, TrackingProps } from 'mweb/common/tracking/withTracking';
import { RootState } from 'mweb/common/reducers/root';
import {
  ExperimentNames,
  ExperimentGroups,
} from 'mweb/common/actions/experiments';
import { Experiment, Treatment } from 'mweb/common/containers/experiment';

if (process.env.BROWSER) {
  require('./openInApp.sass');
}

export const INTERACTION_CONTENT_OPEN_IN_APP = 'open_in_app';

interface OpenInAppStateProps {
  os: OS;
  location: Location;
  referrer: string;
  branchDeeplinkURL: string | undefined;
}

export interface OpenInAppOwnProps {
  fullWidth?: boolean;
  path: string;
  callToActionIdentifier: string;
  medium: string;
}

export interface OpenInAppProps
  extends OpenInAppStateProps,
    OpenInAppOwnProps,
    TrackingProps {}

interface OpenInAppButtonOpts {
  branchEnabled: boolean;
}

export class OpenInAppBase extends React.Component<OpenInAppProps> {
  static displayName: string = 'OpenInApp';

  isInteractive(): boolean {
    return !!this.props.os && this.props.os !== OS.Unknown;
  }

  render(): JSX.Element {
    return (
      <Experiment experimentName={ExperimentNames.Branch}>
        <Treatment group={ExperimentGroups.Treatment}>
          {this.openInAppButton({ branchEnabled: true })}
        </Treatment>
        <Treatment group={ExperimentGroups.Control}>
          {this.openInAppButton({ branchEnabled: false })}
        </Treatment>
      </Experiment>
    );
  }

  private openInAppButton({
    branchEnabled,
  }: OpenInAppButtonOpts): JSX.Element | false {
    const link =
      branchEnabled && this.props.os === OS.iOS
        ? this.props.branchDeeplinkURL
        : this.deeplinkURL;

    return (
      !!link && (
        <Layout className="open-in-app">
          <Button
            linkTo={link}
            fullWidth={this.props.fullWidth}
            onClick={this.props.trackClick({
              interactionContent: INTERACTION_CONTENT_OPEN_IN_APP,
            })}
          >
            <FormattedMessage
              id="open-in-app--message"
              defaultMessage="Open in App"
            />
          </Button>
        </Layout>
      )
    );
  }

  private get deeplinkURL(): string {
    return buildDeepLinkURL({
      path: this.props.path,
      referrer: this.props.referrer,
      appStoreOpts: {
        os: this.props.os,
        campaign: this.props.location,
        callToActionIdentifier: this.props.callToActionIdentifier,
      },
    });
  }
}

function mapStateToProps(state: RootState): OpenInAppStateProps {
  return {
    os: state.device.os,
    location: state.app.location,
    referrer: state.app.url,
    branchDeeplinkURL: state.app.branchDeeplinkURL,
  };
}

export const OpenInApp = connect(mapStateToProps)(
  withTracking(withLatencyTracker(OpenInAppBase)),
) as React.ComponentClass<OpenInAppOwnProps>;
