import * as React from 'react';
import { connect } from 'react-redux';
import { bindActionCreators, Dispatch } from 'redux';
import { Location as RouterLocation } from 'history';
import {
  AlignItems,
  Display,
  JustifyContent,
  Layout,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
  ZIndex,
} from 'twitch-core-ui';

import {
  RootState,
  isFixedTopNav,
  showOpenInApp,
  getCurrentChannel,
} from 'mweb/common/reducers/root';
import { OpenInApp } from 'mweb/common/containers/openInApp';
import TopNavMenu from 'mweb/common/components/topNavMenu';
import { buildLandingPagePath } from 'mweb/common/utils/pathBuilders';
import { Location } from 'mweb/common/reducers/app';
import { navigationDeclineMobileWebExperience } from 'mweb/common/actions/navigation';
import { withLatencyTracker } from 'mweb/common/latency/withLatencyTracker';
import { extendInteractionMedium } from 'mweb/common/tracking/extendInteractionMedium';
import { ClientOnly } from 'mweb/common/containers/clientOnly';
import { ChannelDetails } from 'mweb/common/reducers/data/channels';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

if (process.env.BROWSER) {
  require('./topNav.sass');
}

export const TOP_NAV_MEDIUM = 'top_nav';
export const CALL_TO_ACTION_IDENTIFIER = TOP_NAV_MEDIUM;
const INTERACTION_MEDIUM = TOP_NAV_MEDIUM;
export const INTERACTION_CONTENT_LOGO = 'logo';

export interface TopNavStateProps {
  location: Location;
  isFixed: boolean;
  currentChannel: ChannelDetails | undefined;
  showOpenInApp: boolean;
}

export interface TopNavDispatchProps {
  switchToDesktop: React.EventHandler<React.MouseEvent<HTMLButtonElement>>;
}

export interface TopNavOwnProps {
  routerLocation: RouterLocation;
}

export interface TopNavProps
  extends TopNavStateProps,
    TopNavDispatchProps,
    TopNavOwnProps {}

export class TopNav extends React.Component<TopNavProps, {}> {
  static displayName: string = 'TopNav';

  isInteractive(): boolean {
    return true;
  }

  render(): JSX.Element | null {
    if (this.props.location === Location.Upsell) {
      return null;
    }
    return (
      <StyledLayout
        className="mw-top-nav"
        elevation={2}
        position={this.props.isFixed ? Position.Fixed : undefined}
        display={Display.Flex}
        justifyContent={JustifyContent.Between}
        alignItems={AlignItems.Center}
        zIndex={ZIndex.Above}
        attachTop
        fullWidth
      >
        <TrackedLink
          to={buildLandingPagePath()}
          interactionContent={INTERACTION_CONTENT_LOGO}
          className="mw-top-nav__logo"
        >
          <SVG asset={SVGAsset.LogoTwitch} width={93} height={32} />
        </TrackedLink>
        <ClientOnly>
          <Layout
            className="mw-top-nav__toolbar"
            display={Display.Flex}
            justifyContent={JustifyContent.End}
            alignItems={AlignItems.Center}
          >
            {this.props.showOpenInApp ? (
              <OpenInApp
                path={this.props.routerLocation.pathname}
                callToActionIdentifier={CALL_TO_ACTION_IDENTIFIER}
                medium={TOP_NAV_MEDIUM}
              />
            ) : null}
            <TopNavMenu
              switchToDesktop={this.props.switchToDesktop}
              currentChannel={this.props.currentChannel}
            />
          </Layout>
        </ClientOnly>
      </StyledLayout>
    );
  }
}

function mapStateToProps(state: RootState): TopNavStateProps {
  return {
    location: state.app.location,
    isFixed: isFixedTopNav(state),
    showOpenInApp: showOpenInApp(state),
    currentChannel: getCurrentChannel(state),
  };
}

function mapDispatchToProps(
  dispatch: Dispatch<RootState>,
): TopNavDispatchProps {
  return bindActionCreators(
    {
      switchToDesktop: navigationDeclineMobileWebExperience,
    },
    dispatch,
  );
}

export default connect(mapStateToProps, mapDispatchToProps)(
  extendInteractionMedium(withLatencyTracker(TopNav), INTERACTION_MEDIUM),
) as React.ComponentClass<TopNavOwnProps>;
