import * as React from 'react';
import { connect } from 'react-redux';
import { bindActionCreators, Dispatch } from 'redux';
import Helmet from 'react-helmet';
import { FormattedMessage } from 'react-intl';
import { RouteComponentProps } from 'react-router-dom';
import { StyledLayout, Background } from 'twitch-core-ui';

import { upsellPageTransition } from 'mweb/common/actions/pages/upsell';
import { RootState } from 'mweb/common/reducers/root';
import { TwitchLogo } from 'mweb/common/components/twitchLogo';
import { DeclineUpsell } from 'mweb/common/components/declineUpsell';
import { OpenInApp } from 'mweb/common/containers/openInApp';
import { navigationDeclineUpsell } from 'mweb/common/actions/navigation';
import HeadContainer from 'mweb/common/containers/head';
import { withLatencyTracker } from 'mweb/common/latency/withLatencyTracker';

if (process.env.BROWSER) {
  require('./upsell.sass');
}

const CALL_TO_ACTION_IDENTIFIER = 'primary_button';
const UPSELL_MEDIUM = 'upsell';

export interface UpsellDispatchProps {
  transition: () => void;
  declineUpsell: React.EventHandler<React.MouseEvent<HTMLAnchorElement>>;
}

export interface UpsellOwnProps extends RouteComponentProps<void> {}

export interface UpsellProps extends UpsellDispatchProps, UpsellOwnProps {}

class Upsell extends React.Component<UpsellProps, {}> {
  static displayName: string = 'Upsell';

  isInteractive(): boolean {
    return true;
  }

  componentDidMount(): void {
    this.props.transition();
  }

  render(): JSX.Element {
    return (
      <div className="upsell-page">
        <Helmet>
          <title>Get Twitch!</title>
        </Helmet>
        <HeadContainer path={this.props.location.pathname} />
        <div className="upsell__header">
          <TwitchLogo />
          <p className="upsell__pitch">
            <FormattedMessage
              id="upsell--message"
              defaultMessage="Watch and chat with your favorite broadcasters."
            />
          </p>
        </div>
        <picture className="upsell__image">
          <source
            srcSet={`
              /static/images/upsell-phone.jpg 1x,
              /static/images/upsell-phone@2x.jpg 2x,
              /static/images/upsell-phone@3x.jpg 3x
            `}
            media={`
              (min-width: 540px) and (orientation: portrait), (min-width: 813px) and (orientation: landscape)
            `}
          />
          <img
            className="upsell__default-image"
            src="/static/images/upsell-android.jpg"
            srcSet="/static/images/upsell-android.jpg 1x, /static/images/upsell-android@2x.jpg 2x, /static/images/upsell-android@3x.jpg 3x"
          />
        </picture>
        <footer className="upsell__footer">
          <StyledLayout background={Background.Accent}>
            <OpenInApp
              fullWidth={true}
              path={this.props.location.pathname}
              callToActionIdentifier={CALL_TO_ACTION_IDENTIFIER}
              medium={UPSELL_MEDIUM}
            />
          </StyledLayout>
          <DeclineUpsell onClick={this.props.declineUpsell} />
        </footer>
      </div>
    );
  }
}

function mapDispatchToProps(
  dispatch: Dispatch<RootState>,
): UpsellDispatchProps {
  return bindActionCreators(
    {
      transition: upsellPageTransition,
      declineUpsell: navigationDeclineUpsell,
    },
    dispatch,
  );
}

export default connect(undefined, mapDispatchToProps)(
  withLatencyTracker(Upsell),
) as React.ComponentClass<UpsellOwnProps>;
