import {
  getSpadeDataWithTime,
  SpadeDataWithTime,
} from 'mweb/common/selectors/tracking/base';
import {
  RootState,
  getCurrentChannel,
  getCurrentVODForVODViewer,
} from 'mweb/common/reducers/root';
import { getChannelDetails } from 'mweb/common/selectors/data/channels';
import { Location } from 'mweb/common/reducers/app';
import { VideoType } from 'mweb/common/reducers/data/baseVideoDetails';

export const SUBSCRIBE_BUTTON_ACTION = 'mobile_web_redirect';

export enum SubscribeLocation {
  Channel = 'channel_page',
  ChannelProfile = 'channel_profile_page',
  VOD = 'vod_page',
}

export interface SubscribeButtonSpadeData extends SpadeDataWithTime {
  action: typeof SUBSCRIBE_BUTTON_ACTION;
  can_subscribe: false;
  channel?: string;
  channel_id?: number;
  has_prime: false;
  has_sub_credit: false;
  is_menu_open: false;
  is_subscribed: false;
  logged_in: false;
  modal: false;
  show_prime_content: false;
  show_resub: false;
  sub_location?: SubscribeLocation;
}

interface ChannelSubscribeButtonSpadeData extends SubscribeButtonSpadeData {
  channel: string;
  channel_id: number;
  sub_location: SubscribeLocation.Channel;
}

interface ChannelProfileSubscribeButtonSpadeData
  extends SubscribeButtonSpadeData {
  channel: string;
  channel_id: number;
  sub_location: SubscribeLocation.ChannelProfile;
}

interface VODSubscribeButtonSpadeData extends SubscribeButtonSpadeData {
  channel: string | undefined;
  channel_id: number | undefined;
  sub_location: SubscribeLocation.VOD;
  vod_id: number | undefined;
  vod_type: VideoType | undefined;
}

export function getBaseSubscribeButtonSpadeData(
  state: RootState,
): SubscribeButtonSpadeData {
  return {
    ...getSpadeDataWithTime(state),
    action: SUBSCRIBE_BUTTON_ACTION,
    can_subscribe: false,
    has_prime: false,
    has_sub_credit: false,
    is_menu_open: false,
    is_subscribed: false,
    logged_in: false,
    modal: false,
    show_prime_content: false,
    show_resub: false,
  };
}

export function getChannelSubscribeButtonSpadeData(
  state: RootState,
): ChannelSubscribeButtonSpadeData | undefined {
  const channel = getCurrentChannel(state);
  if (!channel) {
    return undefined;
  }

  return {
    ...getBaseSubscribeButtonSpadeData(state),
    sub_location: SubscribeLocation.Channel,
    channel: channel.name,
    channel_id: parseInt(channel.id),
  };
}

export function getChannelProfileSubscribeButtonSpadeData(
  state: RootState,
): ChannelProfileSubscribeButtonSpadeData | undefined {
  const channel = getCurrentChannel(state);
  if (!channel) {
    return undefined;
  }

  return {
    ...getBaseSubscribeButtonSpadeData(state),
    sub_location: SubscribeLocation.ChannelProfile,
    channel: channel.name,
    channel_id: parseInt(channel.id),
  };
}

export function getVODSubscribeButtonSpadeData(
  state: RootState,
): VODSubscribeButtonSpadeData {
  const vod = getCurrentVODForVODViewer(state);
  const channel = vod && getChannelDetails(state, vod.channel);
  return {
    ...getBaseSubscribeButtonSpadeData(state),
    sub_location: SubscribeLocation.VOD,
    channel: channel && channel.name,
    channel_id: channel && parseInt(channel.id),
    vod_id: vod && parseInt(vod.id),
    vod_type: vod && vod.videoType,
  };
}

export function getSubscribeButtonData(
  state: RootState,
): SubscribeButtonSpadeData | undefined {
  switch (state.app.location) {
    case Location.Channel:
      return getChannelSubscribeButtonSpadeData(state);
    case Location.ChannelProfile:
      return getChannelProfileSubscribeButtonSpadeData(state);
    case Location.VOD:
      return getVODSubscribeButtonSpadeData(state);
    default:
      return undefined;
  }
}
