import expect from 'expect';
import fetchMock from 'fetch-mock';
import { convertUserAgentToBrowserAndOS } from 'src/common/utils/convertUserAgentToBrowserAndOS';

import {
  mockSpadeRotator,
  mockSpadeRotatorError,
  SPADE_TEST_URI,
} from 'mtest/fetchMocks/spadeRotator';
import createTestStore from 'mtest/helpers/createTestStore';

import {
  PLATFORM_SWITCHED_TO_CLIENT_ACTION_TYPE,
  PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE,
  platformSwitchToClient,
  platfromSwitchToServer,
} from 'mweb/common/actions/platform';
import { SPADE_URI_FALLBACK } from 'mweb/common/config/spadeRotator';

describe('platform actions', () => {
  describe('platfromSwitchToServer', () => {
    // must be first because of caching
    it('falls back to default spade uri on request failure', () => {
      const store = createTestStore();
      mockSpadeRotatorError();

      return store
        .dispatch(platfromSwitchToServer({ gitHash: 'cornedbeef' }))
        .then(() => {
          const actions = store.getActions();
          const switchedToServerAction = actions.shift();
          expect(switchedToServerAction.type).toEqual(
            PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE,
          );
          expect(switchedToServerAction.payload).toEqual({
            gitHash: 'cornedbeef',
            spadeURI: SPADE_URI_FALLBACK,
          });

          expect(fetchMock.done()).toEqual(true);
          fetchMock.restore();
        });
    });

    it('fetches the URI when there is no cache', () => {
      const store = createTestStore();
      mockSpadeRotator();

      return store
        .dispatch(platfromSwitchToServer({ gitHash: 'cornedbeef' }))
        .then(() => {
          const actions = store.getActions();
          const switchedToServerAction = actions.shift();
          expect(switchedToServerAction.type).toEqual(
            PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE,
          );
          expect(switchedToServerAction.payload).toEqual({
            gitHash: 'cornedbeef',
            spadeURI: SPADE_TEST_URI,
          });

          expect(fetchMock.done()).toEqual(true);
          fetchMock.restore();
        });
    });

    it('reuses the cached spade URI data and does not re-request', () => {
      const store = createTestStore();
      mockSpadeRotator();

      return store
        .dispatch(platfromSwitchToServer({ gitHash: 'cornedbeef' }))
        .then(() => {
          const actions = store.getActions();
          const switchedToServerAction = actions.shift();
          expect(switchedToServerAction.type).toEqual(
            PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE,
          );
          expect(switchedToServerAction.payload).toEqual({
            gitHash: 'cornedbeef',
            spadeURI: SPADE_TEST_URI,
          });

          expect(fetchMock.done()).toEqual(false); // no requests made
          fetchMock.restore();
        });
    });
  });

  describe('platformSwitchToClient', () => {
    it('assigns the proper device ID, session ID, OS, and browser', () => {
      const userAgent = 'iPad';
      const action = platformSwitchToClient({
        userAgent,
      });
      expect(action.type).toEqual(PLATFORM_SWITCHED_TO_CLIENT_ACTION_TYPE);
      expect(action.payload).toEqual({
        ...convertUserAgentToBrowserAndOS(userAgent),
        netInfo: {
          bandwidth: undefined,
          bandwidth_max: undefined,
          mobile_connection_type: undefined,
          round_trip_time: undefined,
          effective_mobile_connection_type: undefined,
        },
      });
    });
  });
});
