import { stub } from 'sinon';
import expect from 'expect';
import { InjectedIntl } from 'react-intl';
import { CardBody, Layout, AspectRatio } from 'twitch-core-ui';
import { FormattedMessage } from 'react-intl';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { testClickTracking } from 'mtest/helpers/tracking';
import {
  BoxArtCardBase as BoxArtCard,
  BoxArtCardProps,
} from 'mweb/common/components/boxArtCard';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

const DEFAULT_PROPS: BoxArtCardProps = {
  title: 'game title',
  imageTemplate: 'https://{width}/{height}',
  isLatencyCritical: false,
  viewCount: 1230,
  srcSetSizes: [],
  interactionContent: 'interactionContent',
  intl: ({
    formatNumber: stub().returns('1,230'),
    formatMessage: stub().returns('FormattedText'),
  } as any) as InjectedIntl,
};

describe('<BoxArtDirectoryItem />', () => {
  const subject = partialPropShallowWrapper(BoxArtCard, DEFAULT_PROPS);

  it('renders game title', () => {
    expect(
      subject()
        .find(CardBody)
        .find(Layout)
        .childAt(0)
        .shallow()
        .text(),
    ).toEqual(DEFAULT_PROPS.title);
  });

  it('renders viewer count', () => {
    expect(
      subject()
        .find(CardBody)
        .find(Layout)
        .find(FormattedMessage),
    ).toHaveProp('values', {
      viewCount: '1,230',
    });
  });

  it('passes down props to CardImageWrapper', () => {
    expect(subject().find(CardImageWrapper)).toHaveProps({
      latencyTracked: DEFAULT_PROPS.isLatencyCritical,
      aspect: AspectRatio.BoxArt,
      alt: DEFAULT_PROPS.title,
      sizes: DEFAULT_PROPS.srcSetSizes,
    });
  });

  testClickTracking({
    title: 'tracks clicks on box art card',
    expectedPayload: {
      interactionContent: DEFAULT_PROPS.interactionContent,
      interactionTargetPath: '/directory/game/game%20title',
    },
    clickTargetBuilder: () => subject().find(TrackedLink),
  });
});
