import { stub } from 'sinon';
import expect from 'expect';

import { CardBody, Text, Avatar, AspectRatio } from 'twitch-core-ui';
import { InjectedIntl, FormattedMessage } from 'react-intl';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { testClickTracking } from 'mtest/helpers/tracking';

import {
  ChannelThumbnailCardBase as ChannelThumbnailCard,
  ChannelThumbnailCardProps,
} from 'mweb/common/components/channelThumbnailCard';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

describe('<ThumbnailDirectoryItem />', () => {
  const DEFAULT_PROPS: ChannelThumbnailCardProps = {
    imageTemplate: 'https://{width}/{height}',
    name: 'alec',
    displayName: 'Big Alec',
    title: 'A title',
    gameName: 'Some Game',
    viewCount: 1230,
    isLatencyCritical: false,
    logoURL: 'https://alec-lee.com',
    srcSetSizes: [],
    interactionContent: 'interactionContent',
    intl: ({
      formatNumber: stub().returns('1,230'),
      formatMessage: stub().returns('FormattedText'),
    } as any) as InjectedIntl,
  };

  const subject = partialPropShallowWrapper(
    ChannelThumbnailCard,
    DEFAULT_PROPS,
  );

  it('renders title, display name, and game name', () => {
    const card = subject();
    expect(
      card
        .find(CardBody)
        .find(Text)
        .at(0)
        .shallow()
        .text(),
    ).toEqual(DEFAULT_PROPS.title);

    expect(
      card
        .find(CardBody)
        .find(Text)
        .at(1)
        .shallow()
        .text(),
    ).toEqual(DEFAULT_PROPS.displayName);

    expect(
      card
        .find(CardBody)
        .find(Text)
        .at(2)
        .shallow()
        .text(),
    ).toEqual(DEFAULT_PROPS.gameName);
  });

  it('renders viewer count', () => {
    expect(
      subject()
        .find('.channel-thumbnail-card__preview__viewers')
        .find(FormattedMessage),
    ).toHaveProp('values', { viewCount: '1,230' });
  });

  it('renders channel avatar', () => {
    expect(subject().find(Avatar)).toHaveProp('src', DEFAULT_PROPS.logoURL);
  });

  it('passes down props to CardImageWrapper', () => {
    expect(subject().find(CardImageWrapper)).toHaveProps({
      latencyTracked: DEFAULT_PROPS.isLatencyCritical,
      aspect: AspectRatio.Aspect16x9,
      alt: DEFAULT_PROPS.gameName,
      sizes: DEFAULT_PROPS.srcSetSizes,
    });
  });

  testClickTracking({
    title: 'tracks clicks on channel thumbnail card',
    expectedPayload: {
      interactionContent: DEFAULT_PROPS.interactionContent,
      interactionTargetPath: '/alec',
    },
    clickTargetBuilder: () => subject().find(TrackedLink),
  });
});
