import expect from 'expect';
import { FormattedMessage } from 'react-intl';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { testClickTracking, trackClickMock } from 'mtest/helpers/tracking';

import {
  DiscoverMoreBase as DiscoverMore,
  DiscoverMoreProps,
  INTERACTION_CONTENT_CHANNELS,
  INTERACTION_CONTENT_GAMES,
} from 'mweb/common/components/discoverMore';
import {
  buildChannelDirectoryPathFromDecodedGame,
  buildGameDirectoryPath,
} from 'mweb/common/utils/pathBuilders';
import { ALL_CHANNELS } from 'mweb/common/reducers/data/channels';

describe('<DiscoverMore />', () => {
  const DEFAULT_PROPS: DiscoverMoreProps = {
    game: undefined,
    trackClick: trackClickMock(),
  };

  const subject = partialPropShallowWrapper(DiscoverMore, DEFAULT_PROPS);

  describe('with a game', () => {
    const game = 'Pets: Cats/Dogs';

    describe('channel directory button', () => {
      it('formats a message with the right ID', () => {
        expect(
          subject({ game })
            .find('.discover-more__channels')
            .find(FormattedMessage)
            .prop('id'),
        ).toEqual('discover-more--discover-more-of-game');
      });

      it('formats a message with the right game', () => {
        expect(
          subject({ game })
            .find('.discover-more__channels')
            .find(FormattedMessage)
            .prop('values'),
        ).toEqual({ game });
      });

      it('generates the correct link, handling slashes and not converting colons', () => {
        expect(
          subject({ game })
            .find('.discover-more__channels')
            .prop('to'),
        ).toEqual('/directory/game/Pets:%20Cats%2FDogs');
      });
    });

    describe('all games button', () => {
      it('renders the all games button text', () => {
        expect(
          subject({ game })
            .find('.discover-more__games')
            .find(FormattedMessage)
            .prop('id'),
        ).toEqual('discover-more--discover-more-games');
      });
    });

    testClickTracking([
      {
        title: 'tracks clicks on all channels button',
        expectedPayload: {
          interactionContent: INTERACTION_CONTENT_CHANNELS,
          interactionTargetPath: buildChannelDirectoryPathFromDecodedGame(game),
        },
        clickTargetBuilder: () =>
          subject({ game }).find('.discover-more__channels'),
      },
      {
        title: 'tracks clicks on all games button',
        expectedPayload: {
          interactionContent: INTERACTION_CONTENT_GAMES,
          interactionTargetPath: buildGameDirectoryPath(),
        },
        clickTargetBuilder: () =>
          subject({ game }).find('.discover-more__games'),
      },
    ]);
  });

  describe('without a game', () => {
    describe('channel directory button', () => {
      it('formats a message with the right ID', () => {
        expect(
          subject()
            .find('.discover-more__channels')
            .find(FormattedMessage)
            .prop('id'),
        ).toEqual('discover-more--discover-all');
      });

      it('generates the correct link', () => {
        expect(
          subject()
            .find('.discover-more__channels')
            .prop('to'),
        ).toEqual('/directory/all');
      });
    });

    describe('all games button', () => {
      it('renders the all games button text', () => {
        expect(
          subject()
            .find('.discover-more__games')
            .find(FormattedMessage)
            .prop('id'),
        ).toEqual('discover-more--discover-more-games');
      });
    });

    testClickTracking([
      {
        title: 'tracks clicks on all channels button',
        expectedPayload: {
          interactionContent: INTERACTION_CONTENT_CHANNELS,
          interactionTargetPath: buildChannelDirectoryPathFromDecodedGame(
            ALL_CHANNELS,
          ),
        },
        clickTargetBuilder: () => subject().find('.discover-more__channels'),
      },
      {
        title: 'tracks clicks on all games button',
        expectedPayload: {
          interactionContent: INTERACTION_CONTENT_GAMES,
          interactionTargetPath: buildGameDirectoryPath(),
        },
        clickTargetBuilder: () => subject().find('.discover-more__games'),
      },
    ]);
  });
});
