import expect from 'expect';
import { stub, SinonStub } from 'sinon';
import { InjectedIntl, FormattedMessage } from 'react-intl';
import { Stat, SVG, SVGAsset, Text } from 'twitch-core-ui';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { testClickTracking } from 'mtest/helpers/tracking';

import {
  FeaturedContentCardBase as FeaturedContentCard,
  FeaturedContentCardProps,
  INTERACTION_CONTENT_THUMBNAIL,
  INTERACTION_CONTENT_TITLE,
} from 'mweb/common/components/featuredContentCard';
import { buildVODPath } from 'mweb/common/utils/pathBuilders';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';
import { CardImageWrapper } from 'mweb/common/components/more-ui/cardImageWrapper';
import { BaseVideoDetails } from 'mweb/common/reducers/data/baseVideoDetails';

describe('<FeaturedContentCard/>', () => {
  const VOD_ID = '123';

  const DEFAULT_PROPS: FeaturedContentCardProps = {
    hostingChannel: '',
    content: {
      id: VOD_ID,
      formattedLength: '1:23:45',
      game: 'Game',
      thumbnailURL: '//preview.img',
      title: 'Playing the Game',
      viewCount: 0,
    } as BaseVideoDetails,
    intl: ({
      formatNumber: stub().returnsArg(0),
      formatMessage: stub().returns('Text'),
    } as any) as InjectedIntl,
  };
  const content = DEFAULT_PROPS.content as BaseVideoDetails;

  const subject = partialPropShallowWrapper(FeaturedContentCard, DEFAULT_PROPS);

  it('renders the content card when id is present', () => {
    const VOD_LINK = buildVODPath(VOD_ID);

    const card = subject();

    expect(card.find(FormattedMessage).prop('id')).toEqual(
      'channel-profile-most-recent-broadcast',
    );

    const image = card.find('.mw-featured-content-card__image');
    expect(image.find(TrackedLink).prop('to')).toEqual(VOD_LINK);

    const thumbnail = image.find(CardImageWrapper);
    expect(thumbnail).toHaveProps({
      latencyTracked: true,
      alt: content.title,
    });

    expect(image.find(SVG).prop('asset')).toEqual(SVGAsset.Play);

    expect(
      image
        .find('.mw-featured-content-card__views')
        .find(Stat)
        .prop('icon'),
    ).toEqual(SVGAsset.GlyphViews);
    expect(
      (DEFAULT_PROPS.intl.formatMessage as SinonStub).getCall(0).args[0].id,
    ).toEqual('views');

    const length = image.find('.mw-featured-content-card__time').find(Stat);
    expect(length.prop('icon')).toEqual(SVGAsset.GlyphLength);
    expect(length.prop('value')).toEqual(content.formattedLength);
    expect(
      (DEFAULT_PROPS.intl.formatMessage as SinonStub).getCall(1).args[0].id,
    ).toEqual('video-length');

    const details = card.find('.mw-featured-content-card__details');
    expect(details.find(TrackedLink).prop('to')).toEqual(VOD_LINK);
    expect(
      details
        .find(Text)
        .first()
        .childAt(0)
        .text(),
    ).toEqual(content.title);
    expect(
      details
        .find(Text)
        .last()
        .childAt(0)
        .text(),
    ).toEqual(content.game);
  });

  describe('view stats', () => {
    it('renders when not given viewCount', () => {
      const views = subject()
        .find('.mw-featured-content-card__views')
        .find(Stat);
      expect(views.prop('value')).toEqual(0);
    });

    it('renders when given viewCount', () => {
      const VIEWS = 456;

      const views = subject({ content: { id: VOD_ID, viewCount: VIEWS } })
        .find('.mw-featured-content-card__views')
        .find(Stat);
      expect(views.prop('value')).toEqual(VIEWS);
    });
  });

  testClickTracking([
    {
      title: 'tracks clicks on the thumbnail',
      expectedPayload: {
        interactionContent: INTERACTION_CONTENT_THUMBNAIL,
        interactionTargetPath: buildVODPath(VOD_ID),
      },
      clickTargetBuilder: () =>
        subject()
          .find('.mw-featured-content-card__image')
          .find(TrackedLink),
    },
    {
      title: 'tracks clicks on the title',
      expectedPayload: {
        interactionContent: INTERACTION_CONTENT_TITLE,
        interactionTargetPath: buildVODPath(VOD_ID),
      },
      clickTargetBuilder: () =>
        subject()
          .find('.mw-featured-content-card__details')
          .find(TrackedLink),
      clickTargetTrackingIndex: 1,
    },
  ]);
});
