import expect from 'expect';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import {
  testClickTracking,
  testClickTrackingInsideFormattedMessage,
  trackClickMock,
} from 'mtest/helpers/tracking';

import {
  InfoBox,
  InfoBoxProps,
  INTERACTION_CONTENT_LOGO,
  INTERACTION_CONTENT_GAME_NAME,
  INTERACTION_CONTENT_HOSTED_CHANNEL,
} from 'mweb/common/components/infoBox';
import { buildChannelProfilePath } from 'mweb/common/utils/pathBuilders';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

describe('<InfoBox />', () => {
  const DEFAULT_PROPS: InfoBoxProps = {
    channelName: 'mycat',
    channelDisplayName: 'MyCat',
    logoUrl: 'http://pets.logo',
    game: 'Pets',
    hostedChannel: '',
    hostedChannelDisplayName: '',
    vodTitle: '',
    trackClick: trackClickMock(),
  };

  const subject = partialPropShallowWrapper(InfoBox, DEFAULT_PROPS);

  describe('basics', () => {
    it('renders expected constant classes', () => {
      const infoBox = subject({});
      expect(infoBox.hasClass('info-box')).toEqual(true);
      expect(infoBox.find('.info-box--metadata').length).toEqual(1);
    });
  });

  describe('logo', () => {
    it('renders when logoUrl is present', () => {
      const infoBox = subject({});
      expect(infoBox.find('.info-box--logo-container').length).toEqual(1);
      expect(infoBox.find('img').prop('src')).toEqual(DEFAULT_PROPS.logoUrl);
      expect(infoBox.find('img').prop('alt')).toEqual(
        DEFAULT_PROPS.channelDisplayName,
      );
    });

    it('does not render when no logoUrl', () => {
      expect(subject({ logoUrl: '' }).find('img').length).toEqual(0);
    });

    it('links to the channel profile', () => {
      const infoBox = subject({});
      expect(
        infoBox
          .find('.info-box--logo-container')
          .find(TrackedLink)
          .prop('to'),
      ).toEqual(buildChannelProfilePath(DEFAULT_PROPS.channelName!));
    });

    it('links to the channel profile of the hoster, not hosted', () => {
      const infoBox = subject({
        channelName: 'BiggestAndy',
        hostedChannel: 'bigandy',
        hostedChannelDisplayName: 'BigAndy',
      });
      expect(
        infoBox
          .find('.info-box--logo-container')
          .find(TrackedLink)
          .prop('to'),
      ).toEqual(buildChannelProfilePath('BiggestAndy'));
    });
  });

  describe('channel name', () => {
    it('renders by itself when not hosting', () => {
      const infoBox = subject();
      expect(infoBox.find('h2').length).toEqual(1);
      expect(infoBox.find('h2').text()).toEqual(
        DEFAULT_PROPS.channelDisplayName,
      );
    });

    it('renders with hosting information when hosting', () => {
      const infoBox = subject({
        hostedChannel: 'voxel',
        hostedChannelDisplayName: 'Voxel',
      });
      const channelNameInfo = infoBox.find('h2');
      expect(channelNameInfo.length).toEqual(2);
      expect(channelNameInfo.first().text()).toEqual(
        DEFAULT_PROPS.channelDisplayName,
      );
      expect(
        channelNameInfo
          .last()
          .find('FormattedMessage')
          .prop('id'),
      ).toEqual('info-box--hosted-channel-name');
    });
  });

  describe('title', () => {
    it('renders title prop if supplied', () => {
      const infoBox = subject({ vodTitle: 'Doing things' });
      const title = infoBox.find('.info-box--playing-info');
      expect(title.length).toEqual(1);
      expect(title.text()).toEqual('Doing things');
    });

    it('renders directory info if supplied without title', () => {
      const infoBox = subject();
      const title = infoBox.find('.info-box--playing-info');
      expect(title.length).toEqual(1);
      expect(title.find('FormattedMessage').prop('id')).toEqual(
        'info-box--playing-info',
      );
    });

    it('does not render when neither title nor directory is supplied', () => {
      const infoBox = subject({ game: '' });
      expect(infoBox.find('.info-box--playing-info').length).toEqual(0);
    });
  });

  testClickTrackingInsideFormattedMessage([
    {
      title: 'tracks clicks on game',
      key: 'game',
      interactionContent: INTERACTION_CONTENT_GAME_NAME,
      formattedMessage: subject()
        .find('.info-box--playing-info')
        .find('FormattedMessage'),
    },
    {
      title: 'tracks clicks on hosted channel',
      key: 'hostedChannel',
      interactionContent: INTERACTION_CONTENT_HOSTED_CHANNEL,
      formattedMessage: subject({
        hostedChannel: 'bigandy',
        hostedChannelDisplayName: 'BigAndy',
      })
        .find('h2')
        .last()
        .find('FormattedMessage'),
    },
  ]);

  testClickTracking([
    {
      title: 'tracks clicks on logo',
      expectedPayload: {
        interactionContent: INTERACTION_CONTENT_LOGO,
        interactionTargetPath: buildChannelProfilePath(
          DEFAULT_PROPS.channelName!,
        ),
      },
      clickTargetBuilder: () =>
        subject()
          .find('.info-box--logo-container')
          .find(TrackedLink),
    },
  ]);
});
