import expect from 'expect';
import { ShallowWrapper } from 'enzyme';
import { Button } from 'twitch-core-ui';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { trackClickMock, testClickTracking } from 'mtest/helpers/tracking';

import {
  OpenInAppBase as OpenInApp,
  OpenInAppProps,
  INTERACTION_CONTENT_OPEN_IN_APP,
} from 'mweb/common/containers/openInApp';
import { OS } from 'mweb/common/reducers/device';
import { Location } from 'mweb/common/reducers/app';
import { TrackClickHandler } from 'mweb/common/tracking/withTracking';
import { Experiment } from 'mweb/common/containers/experiment';
import { IOS_INTERSTITIAL_BASE_URL } from 'mweb/common/utils/deepLink';

describe('<OpenInApp />', () => {
  const BRANCH_URL = 'www.alec-lee.com';
  const DEFAULT_PROPS: OpenInAppProps = {
    os: OS.iOS,
    location: Location.Upsell,
    path: '/foo/bar',
    callToActionIdentifier: 'pulsing_widget',
    medium: 'medium',
    referrer: 'm.twitch.tv',
    branchDeeplinkURL: undefined,
    trackClick: trackClickMock(),
  };

  const subject = partialPropShallowWrapper(OpenInApp, DEFAULT_PROPS);
  it('renders experiment', () => {
    const openInApp = subject();
    expect(openInApp).toContain(Experiment);
    expect(openInApp.find(Experiment).children().length).toEqual(2);
  });

  it('creates Buttons', () => {
    expect(branchButton({ branchDeeplinkURL: 'www.alec-lee.com' })).toExist();
    expect(button()).toExist();
  });

  it('does not create full-width buttons when fullWidth is false', () => {
    expect(branchButton({ branchDeeplinkURL: 'www.alec-lee.com' })).toHaveProp(
      'fullWidth',
      undefined,
    );
    expect(button()).toHaveProp('fullWidth', undefined);
  });

  it('creates full width buttons when fullWidth is set', () => {
    expect(
      branchButton({
        branchDeeplinkURL: 'www.alec-lee.com',
        fullWidth: true,
      }),
    ).toHaveProp('fullWidth', true);

    expect(
      button({
        fullWidth: true,
      }),
    ).toHaveProp('fullWidth', true);
  });

  it('has the right text', () => {
    expect(
      branchButton({ branchDeeplinkURL: 'www.alec-lee.com' }).find(
        'FormattedMessage',
      ),
    ).toHaveProp('id', 'open-in-app--message');

    expect(button().find('FormattedMessage')).toHaveProp(
      'id',
      'open-in-app--message',
    );
  });

  it('does not render open in app if in branch.io treatment group, on iOS, but no branch url', () => {
    expect(branchButton()).toNotExist();
  });

  it('renders open in app if in branch.io treatment group, on iOS, and have branch url', () => {
    expect(branchButton({ branchDeeplinkURL: BRANCH_URL })).toHaveProp(
      'linkTo',
      BRANCH_URL,
    );
  });

  it('uses mweb created url if in branch.io treatment group but on Android', () => {
    expect(
      branchButton({
        branchDeeplinkURL: BRANCH_URL,
        os: OS.Android,
      }),
    ).toHaveProp(
      'linkTo',
      'intent://m.twitch.tv/foo/bar?referrer=m.twitch.tv' +
        '#Intent;package=tv.twitch.android.app;scheme=twitch;' +
        'S.market_referrer=utm_campaign%3Dmobile_upsell%26utm_source%3Dmwc;' +
        'S.browser_fallback_url=market%3A%2F%2Fdetails%3Fid%3Dtv.twitch.android.app;end;',
    );
  });

  it('uses mweb created url if in branch.io control group', () => {
    expect(
      button({
        branchDeeplinkURL: BRANCH_URL,
        os: OS.Android,
      }),
    ).toHaveProp(
      'linkTo',
      'intent://m.twitch.tv/foo/bar?referrer=m.twitch.tv' +
        '#Intent;package=tv.twitch.android.app;scheme=twitch;' +
        'S.market_referrer=utm_campaign%3Dmobile_upsell%26utm_source%3Dmwc;' +
        'S.browser_fallback_url=market%3A%2F%2Fdetails%3Fid%3Dtv.twitch.android.app;end;',
    );

    const url = `${IOS_INTERSTITIAL_BASE_URL}${DEFAULT_PROPS.path}?c=${DEFAULT_PROPS.location}-${DEFAULT_PROPS.callToActionIdentifier}`;
    expect(
      button({
        branchDeeplinkURL: BRANCH_URL,
        os: OS.iOS,
      }),
    ).toHaveProp('linkTo', url);
  });

  testClickTracking([
    {
      title: 'tracks clicks on the branch button',
      expectedPayload: {
        interactionContent: INTERACTION_CONTENT_OPEN_IN_APP,
      },
      clickTargetBuilder: (trackClick: TrackClickHandler) => {
        return branchButton({
          branchDeeplinkURL: 'www.alec-lee.com',
          trackClick,
        });
      },
    },
    {
      title: 'tracks clicks on the non-branch button',
      expectedPayload: {
        interactionContent: INTERACTION_CONTENT_OPEN_IN_APP,
      },
      clickTargetBuilder: (trackClick: TrackClickHandler) => {
        return button({
          trackClick,
        });
      },
    },
  ]);

  function branchButton(props?: DeepPartial<OpenInAppProps>): ShallowWrapper {
    return subject(props)
      .find(Experiment)
      .children()
      .at(0)
      .find(Button);
  }

  function button(props?: DeepPartial<OpenInAppProps>): ShallowWrapper {
    return subject(props)
      .find(Experiment)
      .children()
      .at(1)
      .find(Button);
  }
});
