import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import expect from 'expect';
import { FormattedMessage } from 'react-intl';
import { Interactable } from 'twitch-core-ui';
import { trackClickMock, testClickTracking } from 'mtest/helpers/tracking';
import { TrackClickHandler } from 'mweb/common/tracking/withTracking';
import { Dropdown } from 'mweb/common/components/buttons/dropdown';
import {
  RemindMeButtonBase as RemindMeButton,
  RemindMeButtonProps,
  RemindMeMenuItemBase as RemindMeMenuItem,
  RemindMeMenuItemProps,
  INTERACTION_CONTENT_REMIND_ME,
} from 'mweb/common/components/remindMeButton';
import { Location } from 'mweb/common/reducers/app';

describe('<RemindMeButton />', () => {
  const DEFAULT_PROPS: RemindMeButtonProps = {
    startTime: new Date('2999-01-01T00:00:00'),
    endTime: new Date('2999-01-01T01:00:00'),
    title: '24 Hour Stream-O-Thon',
    description:
      'I will staire at the screen for 24 hours while you stare at me!',
    isMobileOS: true,
    appLocation: Location.EventDetails,
    handleNotificationRequest: () => undefined,
    trackClick: trackClickMock(),
  };
  const subject = partialPropShallowWrapper(RemindMeButton, DEFAULT_PROPS);

  it('renders a dropdown', () => {
    expect(subject().find(Dropdown).length).toEqual(1);
  });

  describe('the dropdown', () => {
    const dropdown = subject().find(Dropdown);

    it('has the right formatted message', () => {
      expect(dropdown.find(FormattedMessage).prop('id')).toEqual(
        'add-reminder-button--top-level-message',
      );
    });

    it('has a google calendar option', () => {
      const items = dropdown.prop('dropdownItems') as Array<JSX.Element>;
      expect(items[0].key).toEqual('Google Calendar');
    });

    it('has a Apple calendar option', () => {
      const items = dropdown.prop('dropdownItems') as Array<JSX.Element>;
      expect(items[1].key).toEqual('Apple Calendar');
    });
  });

  describe('When the date is in the past', () => {
    it('does not renders a dropdown', () => {
      expect(
        subject({ startTime: new Date(), endTime: new Date() }).find(Dropdown)
          .length,
      ).toEqual(0);
    });
  });

  testClickTracking({
    title: 'tracks clicks on remind me button',
    expectedPayload: {
      interactionContent: INTERACTION_CONTENT_REMIND_ME,
    },
    clickTargetBuilder: (trackClick: TrackClickHandler) =>
      subject({
        trackClick,
      }).find(Dropdown),
  });
});

describe('<RemindMeMenuItem />', () => {
  const DEFAULT_PROPS: RemindMeMenuItemProps = {
    trackClick: trackClickMock(),
    text: 'text',
  };

  const subject = partialPropShallowWrapper(RemindMeMenuItem, DEFAULT_PROPS);

  testClickTracking({
    title: 'tracks clicks on remind me menu item',
    expectedPayload: {
      interactionContent: 'BigAndy',
    },
    clickTargetBuilder: (trackClick: TrackClickHandler) =>
      subject({
        trackClick,
        tracking: { interactionContent: 'BigAndy' },
      }).find(Interactable),
  });
});
