import expect from 'expect';
import { stub } from 'sinon';
import { InjectedIntl } from 'react-intl';
import { Text, ButtonIcon } from 'twitch-core-ui';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';

import {
  ShareSheetBase as ShareSheet,
  ShareSheetProps,
} from 'mweb/common/components/shareSheet';
import { Facebook } from 'mweb/common/components/svg/facebook';
import { Copy } from 'mweb/common/components/svg/copy';
import { Reddit } from 'mweb/common/components/svg/reddit';
import { Twitter } from 'mweb/common/components/svg/twitter';
import { Location } from 'mweb/common/reducers/app';
import { ShareSheetItem } from 'mweb/common/components/shareSheetItem';

describe('<ShareSheet />', () => {
  const DEFAULT_PROPS: ShareSheetProps = {
    header: 'Fun header',
    isVisible: true,
    title: 'A Titular Title',
    appLocation: Location.EventDetails,
    toggleSharingDock: () => undefined,
    handleShare: () => undefined,
    intl: ({
      formatNumber: stub().returnsArg(0),
      formatMessage: stub().returns('FormattedText'),
    } as any) as InjectedIntl,
  };

  const subject = partialPropShallowWrapper(ShareSheet, DEFAULT_PROPS);

  it('does not generate share options if not visible', () => {
    const share = subject({ isVisible: false });
    expect(share).toNotContain(Twitter);
    expect(share).toNotContain(Reddit);
    expect(share).toNotContain(Facebook);
    expect(share).toNotContain(Copy);
  });

  it('creates header', () => {
    const shareHeader = subject().find('.share-sheet__header');
    expect(
      shareHeader
        .find(Text)
        .shallow()
        .text(),
    ).toEqual(DEFAULT_PROPS.header);
    expect(shareHeader).toContain(ButtonIcon);
  });

  it('generates the proper share options', () => {
    const share = subject();

    const twitterLink = share.find(Twitter).closest(ShareSheetItem);
    const twitterLinkText = encodeURIComponent(
      `${DEFAULT_PROPS.title} ${window.location.href}`,
    );
    expect(twitterLink).toHaveProps({
      url: `https://twitter.com/intent/tweet?text=${twitterLinkText}`,
    });

    const redditLink = share.find(Reddit).closest(ShareSheetItem);
    expect(redditLink).toHaveProps({
      url: `http://www.reddit.com/submit?url=${encodeURIComponent(
        window.location.href,
      )}&title=${encodeURIComponent(DEFAULT_PROPS.title)}`,
    });

    const facebookLink = share.find(Facebook).closest(ShareSheetItem);
    expect(facebookLink).toHaveProps({
      url: `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(
        window.location.href,
      )}`,
    });

    const copy = share.find(Copy).closest(ShareSheetItem);
    expect(copy).toHaveProps({
      url: window.location.href,
      displayName: 'FormattedText',
      isClipboard: true,
    });
  });
});
