import expect from 'expect';
import { match } from 'react-router-dom';
import { Location, History } from 'history';
import { spy } from 'sinon';
import { Redirect } from 'react-router-dom';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';

import { ChannelVideoPlayer } from 'mweb/common/components/videoPlayer/channel';
import InfoBox from 'mweb/common/components/infoBox';
import {
  ChannelViewer,
  ChannelViewerProps,
  ChannelViewerMatchParams,
} from 'mweb/common/containers/channelViewer';
import { BranchJourneyType } from 'branch-io';
import { BranchUpsellExperiment } from 'mweb/common/components/branchUpsell';

describe('<ChannelViewer/>', () => {
  const DEFAULT_PROPS: ChannelViewerProps = {
    liveToVODStreamID: '0',
    liveToVODTargetVODid: '',
    channelDisplayName: 'Voxel',
    channelID: '3',
    logoUrl: 'http://voxel.logo',
    game: 'creative',
    hostedChannel: '',
    hostedChannelDisplayName: '',
    chatMessages: [],
    showChat: undefined,
    sessionID: 'sessionID',
    isChatReady: false,
    transitionToTargetChannel: () => undefined,
    checkLiveToVODStatus: () => undefined,
    switchToDesktop: () => undefined,
    match: {
      params: {
        channel: 'voxel',
      },
    } as match<ChannelViewerMatchParams>,
    location: { pathname: '/voxel', search: '' } as Location,
    history: {} as History,
  };

  const subject = partialPropShallowWrapper(ChannelViewer, DEFAULT_PROPS);

  it('shows the video player and info box for a channel', () => {
    const channel = subject();
    expect(channel.find(Redirect).length).toEqual(0);
    expect(channel.find(InfoBox).length).toEqual(1);
    expect(channel.find(ChannelVideoPlayer).length).toEqual(1);
  });

  it('creates branch upsell', () => {
    const channel = subject();
    expect(channel).toContain(BranchUpsellExperiment);
    expect(channel.find(BranchUpsellExperiment)).toHaveProps({
      delaySecondsUntilUpsell: 120,
      journey: {
        type: BranchJourneyType.Live,
        channel: DEFAULT_PROPS.channelDisplayName,
      },
    });
  });

  describe('LiveToVOD', () => {
    it('calls checkLiveToVODStatus if given a stream ID', () => {
      const checkLiveToVODStatus = spy();
      const liveToVODStreamID = '5';
      const channel = subject({ checkLiveToVODStatus });

      channel.setProps({ liveToVODStreamID });
      (channel.instance() as ChannelViewer).componentDidUpdate(DEFAULT_PROPS);
      expect(checkLiveToVODStatus.called).toEqual(true);
      expect(checkLiveToVODStatus.getCall(0).args).toEqual([
        DEFAULT_PROPS.channelID,
        liveToVODStreamID,
      ]);
    });

    it('renders a redirect when given a liveToVODTargetVODid', () => {
      const liveToVODTargetVODid = '123';
      const redir = subject({ liveToVODTargetVODid }).find(Redirect);
      expect(redir.length).toEqual(1);
      expect(redir.prop('to')).toEqual('/videos/123');
    });
  });
});
