import expect from 'expect';
import { match } from 'react-router-dom';
import { Location, History } from 'history';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { EVENT_MODEL } from 'mtest/fetchMocks/event';
import { CHANNEL_DETAILS } from 'mtest/fetchMocks/channel';
import { GAME_1 } from 'mtest/fetchMocks/games';

import {
  EventDetails,
  EventDetailsProps,
  EventDetailsMatchParams,
} from 'mweb/common/containers/eventDetails';
import { EventBanner } from 'mweb/common/components/eventBanner';
import { EventInfo } from 'mweb/common/components/eventInfo';
import { EventActions } from 'mweb/common/components/eventActions';
import EventPanels from 'mweb/common/components/eventPanels';
import { Location as AppLocation } from 'mweb/common/reducers/app';
import Spinner from 'mweb/common/components/spinner';

describe('<EventDetails />', () => {
  const DEFAULT_PROPS_WITHOUT_EVENT: EventDetailsProps = {
    event: undefined,
    channel: CHANNEL_DETAILS,
    game: GAME_1,
    isMobileOS: true,
    appLocation: AppLocation.EventDetails,
    premiereStatus: undefined,
    video: undefined,
    transitionToEvent: () => undefined,
    handleShare: () => undefined,
    handleNotificationRequest: () => undefined,
    match: {
      params: {
        eventID: EVENT_MODEL.id,
      },
    } as match<EventDetailsMatchParams>,
    location: {
      pathname: `/events/${EVENT_MODEL.id}`,
      search: '',
    } as Location,
    history: {} as History,
  };

  const DEFAULT_PROPS = {
    ...DEFAULT_PROPS_WITHOUT_EVENT,
    event: EVENT_MODEL,
  };

  const subject = partialPropShallowWrapper(EventDetails, DEFAULT_PROPS);

  it('renders spinner if no event', () => {
    expect(
      partialPropShallowWrapper(EventDetails, DEFAULT_PROPS_WITHOUT_EVENT)(),
    ).toContain(Spinner);
  });

  it('does not render spinner if event', () => {
    expect(subject()).toNotContain(Spinner);
  });

  it('renders the event description', () => {
    expect(
      subject({ event: undefined })
        .find('.event-description')
        .shallow()
        .text(),
    ).toEqual(EVENT_MODEL.description);
  });

  it('creates EventBanner with correct props', () => {
    expect(
      subject()
        .find(EventBanner)
        .props(),
    ).toEqual({
      coverImageTemplateURL: EVENT_MODEL.coverImageTemplateURL,
      startTime: new Date(EVENT_MODEL.startTime),
    });
  });

  it('creates EventInfo with correct props', () => {
    expect(
      subject()
        .find(EventInfo)
        .props(),
    ).toEqual({
      startTime: new Date(EVENT_MODEL.startTime),
      endTime: new Date(EVENT_MODEL.endTime),
      title: EVENT_MODEL.title,
      premiereStatus: undefined,
    });
  });
  it('creates EventActions with correct props', () => {
    expect(
      subject()
        .find(EventActions)
        .props(),
    ).toEqual({
      startTime: new Date(EVENT_MODEL.startTime),
      endTime: new Date(EVENT_MODEL.endTime),
      title: EVENT_MODEL.title,
      description: EVENT_MODEL.description,
      isMobileOS: true,
      appLocation: AppLocation.EventDetails,
      handleNotificationRequest: DEFAULT_PROPS.handleNotificationRequest,
      handleShare: DEFAULT_PROPS.handleShare,
      premiereStatus: undefined,
      channelName: CHANNEL_DETAILS.name,
    });
  });

  it('creates EventPanels with correct props', () => {
    expect(
      subject()
        .find(EventPanels)
        .props(),
    ).toEqual({
      logoURL: CHANNEL_DETAILS.logoURL,
      channelDisplayName: CHANNEL_DETAILS.displayName,
      game: GAME_1,
      channelName: CHANNEL_DETAILS.name,
      video: undefined,
    });
  });
});
