import expect from 'expect';
import { Position, SVG, SVGAsset } from 'twitch-core-ui';

import { partialPropShallowWrapper } from 'mtest/helpers/partialPropWrappers';
import { testClickTracking } from 'mtest/helpers/tracking';
import { CHANNEL_DETAILS } from 'mtest/fetchMocks/channel';

import {
  CALL_TO_ACTION_IDENTIFIER,
  TopNav,
  TopNavProps,
  TOP_NAV_MEDIUM,
  INTERACTION_CONTENT_LOGO,
} from 'mweb/common/containers/topNav';
import { Location } from 'mweb/common/reducers/app';
import { buildLandingPagePath } from 'mweb/common/utils/pathBuilders';
import { OpenInApp } from 'mweb/common/containers/openInApp';
import { TrackedLink } from 'mweb/common/tracking/trackedLink';

describe('<TopNav/>', () => {
  const DEFAULT_LOCATION = {
    pathname: '/a/b/c',
    search: '',
    state: {},
    hash: '',
    key: '',
  };

  const DEFAULT_PROPS: TopNavProps = {
    location: Location.Channel,
    switchToDesktop: () => undefined,
    isFixed: false,
    routerLocation: DEFAULT_LOCATION,
    currentChannel: CHANNEL_DETAILS,
    showOpenInApp: true,
  };

  const subject = partialPropShallowWrapper(TopNav, DEFAULT_PROPS);

  it('does not have fixed position by default', () => {
    expect(
      subject()
        .find('.mw-top-nav')
        .prop('position'),
    ).toEqual(undefined);
  });

  it('has fixed position when passed isFixed', () => {
    expect(
      subject({ isFixed: true })
        .find('.mw-top-nav')
        .prop('position'),
    ).toEqual(Position.Fixed);
  });

  it('renders the logo image with a link back to the landing page', () => {
    const logo = subject().find('.mw-top-nav__logo');

    expect(logo.prop('to')).toEqual('/');
    expect(logo.find(SVG).prop('asset')).toEqual(SVGAsset.LogoTwitch);
  });

  it('passes the STD function to the TopNavMenu', () => {
    expect(
      subject({})
        .find('Trackable(TopNavMenu)')
        .prop('switchToDesktop'),
    ).toEqual(DEFAULT_PROPS.switchToDesktop);
  });

  it('passes the right values to the OpenInApp button', () => {
    const openInApp = subject()
      .find('.mw-top-nav__toolbar')
      .find(OpenInApp);

    expect(openInApp.props()).toEqual({
      path: DEFAULT_LOCATION.pathname,
      callToActionIdentifier: CALL_TO_ACTION_IDENTIFIER,
      medium: TOP_NAV_MEDIUM,
    });
  });

  it('Does not add OpenInApp if showOpenInApp is false', () => {
    const openInApp = subject({ showOpenInApp: false })
      .find('.mw-top-nav__toolbar')
      .find(OpenInApp);
    expect(openInApp.length).toEqual(0);
  });

  testClickTracking({
    title: 'tracks click on box art',
    expectedPayload: {
      interactionContent: INTERACTION_CONTENT_LOGO,
      interactionTargetPath: buildLandingPagePath(),
    },
    clickTargetBuilder: () => subject().find(TrackedLink),
  });
});
