import faker from 'faker';

import { CHANNEL, makeFakeChannelGQL } from 'mtest/fetchMocks/channel';
import { mockGQL } from 'mtest/fetchMocks/mockGQL';

import {
  FetchVideoGQL,
  VODDataPayload,
  GQLBroadcastType,
  mapGQLBroadcastTypeToVideoType,
} from 'mweb/common/fetch/vods';
import { VODDetails } from 'mweb/common/reducers/data/vods';
import {
  ChannelDetails,
  ChannelOnlineStatus,
  ChannelSubscribableStatus,
} from 'mweb/common/reducers/data/channels';
import { formatVideoLength } from 'mweb/common/utils/formatVideoLength';
import { VideoType } from 'mweb/common/reducers/data/baseVideoDetails';

export const VOD = '12345';
export const NOT_VOD = '67890';

export function makeFakeFetchVideoGQL(
  id: string,
  channel?: string,
): FetchVideoGQL {
  return {
    data: {
      video: {
        id,
        previewThumbnailURL: faker.image.imageUrl(90, 90),
        owner: makeFakeChannelGQL(channel || CHANNEL),
        title: faker.lorem.words(20),
        viewCount: faker.random.number(),
        lengthSeconds: faker.random.number(),
        broadcastType: GQLBroadcastType.Archive,
        description: faker.lorem.sentence(),
        recordedAt: faker.date.past().toString(),
        createdAt: faker.date.past().toString(),
        game: {
          name: faker.hacker.phrase(),
          id: faker.random.alphaNumeric(),
        },
      },
    },
  };
}

export const VOD_JSON: FetchVideoGQL = makeFakeFetchVideoGQL(VOD);

export const VOD_NOT_FOUND_JSON: FetchVideoGQL = {
  data: {
    video: {
      id: '',
      previewThumbnailURL: '',
      title: null,
      viewCount: null,
      lengthSeconds: null,
      broadcastType: null,
      description: null,
      recordedAt: null,
      createdAt: null,
      game: null,
    },
  },
};

const VIDEO = VOD_JSON.data.video!;
const USER = VIDEO.owner!;

export const VOD_DETAILS: VODDetails = {
  id: VIDEO.id,
  channel: USER.login!,
  thumbnailURL: VIDEO.previewThumbnailURL,
  videoType: mapGQLBroadcastTypeToVideoType(VIDEO.broadcastType),
  description: VIDEO.description!,
  game: VIDEO.game!.name,
  title: VIDEO.title!,
  viewCount: VIDEO.viewCount!,
  length: VIDEO.lengthSeconds!,
  formattedLength: formatVideoLength(VIDEO.lengthSeconds!),
  date: new Date(VIDEO.recordedAt!).valueOf(),
};

export const VOD_HIGHLIGHT_DETAILS: VODDetails = {
  ...VOD_DETAILS,
  videoType: VideoType.Highlight,
};

const CHANNEL_DETAILS: ChannelDetails = {
  displayName: USER.displayName!,
  game: USER.stream!.game!.name,
  hostedChannel: undefined,
  id: USER.id!,
  logoURL: USER.profileImageURL!,
  name: USER.login!,
  onlineStatus: ChannelOnlineStatus.Online,
  preview: USER.stream!.previewImageURL!,
  status: USER.stream!.title!,
  viewerCount: USER.stream!.viewersCount!,
  subscribableStatus: ChannelSubscribableStatus.Unknown,
  description: USER.description!,
  bannerImageURL: USER.bannerImageURL!,
  followerCount: USER.followers!.totalCount,
  lifetimeViewerCount: USER.profileViewCount!,
};

export const VOD_DATA_PAYLOAD: VODDataPayload = {
  vod: VOD_DETAILS,
  channel: CHANNEL_DETAILS,
};

export function mockVOD(): void {
  mockGQL(VOD_JSON);
}

export function mockVODNotFound(): void {
  mockGQL(VOD_NOT_FOUND_JSON);
}
