import expect from 'expect';
import {
  ExperimentOverrideMapping,
  ExperimentGroups,
  ExperimentUUIDs,
} from 'mweb/common/actions/experiments';
import { parseExperimentOverrides } from 'mweb/lambda/experiment-override';

describe('parseExperimentOverrides', () => {
  interface ExperimentOverrideTestCase {
    readonly query: string;
    readonly expected: ExperimentOverrideMapping;
  }

  const TEST_CASES: ExperimentOverrideTestCase[] = [
    {
      query: 'TEST_EXPERIMENT_UUID=treatment',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: 'test_experiment_uuid=treatment',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: 'test_experiment_uuid=TREATMENT',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: 'not_a_uuid=whatever,test_experiment_uuid=TREATMENT',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query:
        'not_a_uuid=whatever,test_experiment_uuid=TREATMENT,also_not_a_uuid=whatever,',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: 'test_experiment_uuid=TREATMENT,also_not_a_uuid=whatever',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: '',
      expected: {},
    },
    {
      query: 'test_experiment_uuid===',
      expected: {},
    },
    {
      query: 'test_experiment_uuid=not_a_group',
      expected: {},
    },
    {
      query: 'test_experiment_uuid=TREATMENT,=whatever',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: 'test_experiment_uuid=TREATMENT,also_not_a_uuid=',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: ',,,also_not_a_uuid=,,test_experiment_uuid=TREATMENT,',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
    {
      query: ',,,also_not_a_uuid=,,test_experiment_uuid==TREATMENT,',
      expected: {},
    },
    {
      query: ',,,==&also_not_a_uuid=,,test_experiment_uuid==TREATMENT,',
      expected: {},
    },
    {
      query: ',,,==&also_not_a_uuid=,,test_experiment_uuid=TREATMENT,',
      expected: {
        [ExperimentUUIDs.Test]: ExperimentGroups.Treatment,
      },
    },
  ];
  TEST_CASES.forEach(({ query, expected }) => {
    it(`parses ${query}`, () => {
      expect(parseExperimentOverrides(query)).toEqual(expected);
    });
  });
});
