import expect from 'expect';

import createTestState from 'mtest/helpers/createTestState';

import { Action } from 'mweb/common/actions/root';
import { tracking } from 'mweb/common/reducers/tracking';
import { PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE } from 'mweb/common/actions/platform';
import { NAVIGATION_UPDATED_LOCATION_ACTION_TYPE } from 'mweb/common/actions/navigation';
import {
  TRACKING_PARAMS_CAPTURED_ACTION_TYPE,
  TRACKING_PARAMS_CLEARED_ACTION_TYPE,
} from 'mweb/common/actions/tracking';
import { Location } from 'mweb/common/reducers/app';

describe('tracking reducer', () => {
  it('sets reasonable defaults', () => {
    expect(tracking(undefined, { type: 'NOOP' })).toEqual({
      medium: undefined,
      content: undefined,
      pageLoadID: '',
      spadeURI: '',
    });
  });

  it('records spade URI from server platform action', () => {
    const action: Action = {
      type: PLATFORM_SWITCHED_TO_SERVER_ACTION_TYPE,
      payload: {
        gitHash: 'cornedbeef',
        spadeURI: '//somewhere.over.the.rainbow',
      },
    };
    expect(tracking(undefined, action).spadeURI).toEqual(
      '//somewhere.over.the.rainbow',
    );
  });

  it('sets the pageLoadID when location is updated', () => {
    const action: Action = {
      type: NAVIGATION_UPDATED_LOCATION_ACTION_TYPE,
      payload: {
        location: Location.Channel,
        intent: 'stream/voxel',
        referrer: 'http://www.twitch.tv/',
      },
    };
    expect(tracking(undefined, action).pageLoadID).toNotEqual('');
  });

  it('resets the pageLoadID with each new location', () => {
    const action: Action = {
      type: NAVIGATION_UPDATED_LOCATION_ACTION_TYPE,
      payload: {
        location: Location.Channel,
        intent: 'stream/voxel',
        referrer: 'http://www.twitch.tv/',
      },
    };
    const initialState = tracking(undefined, action);
    expect(tracking(initialState, action).pageLoadID).toNotEqual(
      initialState.pageLoadID,
    );
  });

  it('clears out tracking params data after TRACKING_PARAMS_CLEARED_ACTION_TYPE', () => {
    const initialState = createTestState({
      tracking: {
        content: 'cheese',
        medium: 'crackers',
      },
    }).tracking;
    const action: Action = {
      type: TRACKING_PARAMS_CLEARED_ACTION_TYPE,
      payload: {},
    };
    const newState = tracking(initialState, action);
    expect(newState.content).toEqual(undefined);
    expect(newState.medium).toEqual(undefined);
  });

  it('handles tracking params with just content', () => {
    const action: Action = {
      type: TRACKING_PARAMS_CAPTURED_ACTION_TYPE,
      payload: {
        tt_content: 'foo',
        tt_medium: undefined,
      },
    };
    const initialState = createTestState({
      tracking: {
        content: 'cheese',
        medium: 'crackers',
      },
    }).tracking;
    const newState = tracking(initialState, action);
    expect(newState.content).toEqual('foo');
    expect(newState.medium).toEqual('crackers');
  });

  it('handles tracking params with just medium', () => {
    const action: Action = {
      type: TRACKING_PARAMS_CAPTURED_ACTION_TYPE,
      payload: {
        tt_content: undefined,
        tt_medium: 'bar',
      },
    };
    const initialState = createTestState({
      tracking: {
        content: 'cheese',
        medium: 'crackers',
      },
    }).tracking;
    const newState = tracking(initialState, action);
    expect(newState.content).toEqual('cheese');
    expect(newState.medium).toEqual('bar');
  });
});
