import expect from 'expect';
import omit from 'lodash-es/omit';

import createTestState from 'mtest/helpers/createTestState';
import { expectEqualWithoutSpadeBaseAndLocation } from 'mtest/helpers/trackingSelectors';

import {
  getLatencyBandwidthSpadeData,
  getLatencyComponentInitSpadeData,
  getLatencySpadeData,
  getLatencyCustomEventSpadeData,
} from 'mweb/common/selectors/tracking/latency';
import { Location } from 'mweb/common/reducers/app';
import {
  PLATFORM_FOR_SPADE,
  SpadeData,
} from 'mweb/common/selectors/tracking/base';

const baseLatencyKeys = Object.keys(
  getLatencySpadeData({ state: createTestState(), epochTime: 1 }),
);
function expectEqualWithLatencyBase(actual: SpadeData, expected: Object): void {
  expect(actual).toIncludeKeys(baseLatencyKeys);
  expect(omit(actual, baseLatencyKeys)).toEqual(expected);
}

describe('tracking latency selectors', () => {
  it('getLatencySpadeData', () => {
    const state = createTestState({
      device: {
        sessionID: 'sessionID',
      },
      app: {
        location: Location.Channel,
      },
    });

    expectEqualWithoutSpadeBaseAndLocation(
      getLatencySpadeData({ state, epochTime: 1000 }),
      {
        benchmark_session_id: 'sessionID',
        client_app: PLATFORM_FOR_SPADE,
        destination: Location.Channel,
        client_time: 1,
      },
    );
  });

  it('getLatencyBandwidthSpadeData none', () => {
    const state = createTestState();

    expectEqualWithoutSpadeBaseAndLocation(
      getLatencyBandwidthSpadeData({ state, epochTime: 1000 }),
      {
        benchmark_session_id: '',
        client_app: PLATFORM_FOR_SPADE,
        destination: Location.Unknown,
        client_time: 1,
      },
    );
  });

  it('getLatencyBandwidthSpadeData common', () => {
    const state = createTestState({
      device: {
        netInfo: {
          round_trip_time: 100,
          bandwidth: 1.5,
        },
      },
    });

    expectEqualWithoutSpadeBaseAndLocation(
      getLatencyBandwidthSpadeData({ state, epochTime: 1000 }),
      {
        benchmark_session_id: '',
        client_app: PLATFORM_FOR_SPADE,
        destination: Location.Unknown,
        client_time: 1,
        round_trip_time: 100,
        bandwidth: 1.5,
      },
    );
  });

  it('getLatencyBandwidthSpadeData all', () => {
    const state = createTestState({
      device: {
        netInfo: {
          round_trip_time: 100,
          bandwidth: 1.5,
          bandwidth_max: 40,
          mobile_connection_type: 'wifi' as ConnectionType,
          effective_mobile_connection_type: '2g' as EffectiveConnectionType,
        },
      },
    });

    expectEqualWithoutSpadeBaseAndLocation(
      getLatencyBandwidthSpadeData({ state, epochTime: 1000 }),
      {
        benchmark_session_id: '',
        client_app: PLATFORM_FOR_SPADE,
        destination: Location.Unknown,
        client_time: 1,
        round_trip_time: 100,
        bandwidth: 1.5,
        bandwidth_max: 40,
        mobile_connection_type: 'wifi' as ConnectionType,
        effective_mobile_connection_type: '2g' as EffectiveConnectionType,
      },
    );
  });

  it('getLatencyComponentInitSpadeData', () => {
    const state = createTestState();

    expectEqualWithLatencyBase(
      getLatencyComponentInitSpadeData({
        state,
        epochTime: 1,
        name: 'a',
        id: 2,
        parentName: 'b',
        parentID: 3,
      }),
      {
        component: 'a',
        component_id: 2,
        parent_component: 'b',
        parent_component_id: 3,
      },
    );
  });

  it('getLatencyCustomEventSpadeData', () => {
    const state = createTestState();

    expectEqualWithLatencyBase(
      getLatencyCustomEventSpadeData({
        state,
        epochTime: 2,
        duration: 123,
        group: 'Group',
        key: 'key',
        label: 'Label',
        isFirstPage: true,
        isDocumentHidden: true,
      }),
      {
        duration: 123,
        group: 'Group',
        key: 'key',
        label: 'Label',
        is_app_launch: true,
        lost_visibility: true,
      },
    );
  });
});
