import expect from 'expect';

import { PageLocalizationManager } from 'mweb/server/pageLocalizationManager';

describe('PageLocalizationManager', () => {
  const ACCEPTED_LANGUAGES = [
    'en-US',
    'en-GB',
    'es-ES',
    'es-MX',
    'zh-TW',
    'fh-fh',
  ];

  describe('construction', () => {
    it('handles properly cased languages', () => {
      const subject = new PageLocalizationManager(
        'en-GB; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('en');
      expect(subject.languageTag).toEqual('en-gb');
    });

    it('defaults when no option is available', () => {
      const subject = new PageLocalizationManager(
        'alien-bogus; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('en');
      expect(subject.languageTag).toEqual('en-us');
    });

    it('defaults when given no accept language', () => {
      const subject = new PageLocalizationManager(
        undefined,
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('en');
      expect(subject.languageTag).toEqual('en-us');
    });

    it('handles all lower-case languages', () => {
      const subject = new PageLocalizationManager(
        'es-mx; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('es');
      expect(subject.languageTag).toEqual('es-mx');
    });

    it('handles all iOS-style language headers', () => {
      const subject = new PageLocalizationManager('es-ES', ACCEPTED_LANGUAGES);
      expect(subject.language).toEqual('es');
      expect(subject.languageTag).toEqual('es-es');
    });

    it('handles Twitch-cookie-style languages (all lowercase, no q)', () => {
      const subject = new PageLocalizationManager('zh-tw', ACCEPTED_LANGUAGES);
      expect(subject.language).toEqual('zh');
      expect(subject.languageTag).toEqual('zh-tw');
    });

    it('handles regionless languages, generally', () => {
      const subject = new PageLocalizationManager(
        'fh; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('fh');
      expect(subject.languageTag).toEqual('fh-fh');
    });

    it('handles regionless Spanish', () => {
      const subject = new PageLocalizationManager(
        'es; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('es');
      expect(subject.languageTag).toEqual('es-es');
    });

    it('handles regionless Chinese', () => {
      const subject = new PageLocalizationManager(
        'zh; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('zh');
      expect(subject.languageTag).toEqual('zh-cn');
    });

    it('handles languages with differing region', () => {
      const subject = new PageLocalizationManager(
        'fh-xx; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.language).toEqual('fh');
      expect(subject.languageTag).toEqual('fh-fh');
    });
  });

  describe('localePolyfillString', () => {
    it('generates the polyfill string', () => {
      const subject = new PageLocalizationManager(
        'en-GB; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.localePolyfillString()).toEqual('Intl.~locale.en');
    });
  });
  describe('reactIntlCode', () => {
    it('generates the polyfill string', () => {
      const subject = new PageLocalizationManager(
        'en-GB; q=0.9',
        ACCEPTED_LANGUAGES,
      );
      expect(subject.reactIntlCode()).toEqual(
        require(`raw-loader!react-intl/locale-data/en`),
      );
    });
  });
});
