import expect from 'expect';

import { Location } from 'mweb/common/reducers/app';
import { OS } from 'mweb/common/reducers/device';
import {
  buildDeepLinkURL,
  IOS_INTERSTITIAL_BASE_URL,
} from 'mweb/common/utils/deepLink';

describe('buildDeepLinkURL', () => {
  describe('without app store opts', () => {
    it('builds with a custom intent', () => {
      expect(
        buildDeepLinkURL({ path: '/directory/bigandy', referrer: 'hi' }),
      ).toEqual('twitch://m.twitch.tv/directory/bigandy?referrer=hi');
    });
  });

  describe('with app store opts', () => {
    it('encodes the referrer', () => {
      expect(
        buildDeepLinkURL({ path: '/a', referrer: 'https://m.twitch.tv?h=i' }),
      ).toEqual(
        'twitch://m.twitch.tv/a?referrer=https%3A%2F%2Fm.twitch.tv%3Fh%3Di',
      );
    });
    it('builds an android deep link', () => {
      expect(
        buildDeepLinkURL({
          path: '/monstercat',
          referrer: 'ref',
          appStoreOpts: {
            os: OS.Android,
            campaign: Location.VOD,
            callToActionIdentifier: 'a_button',
          },
        }),
      ).toEqual(
        'intent://m.twitch.tv/monstercat?referrer=ref#Intent;package=tv.twitch.android.app;scheme=twitch;S.market_referrer=utm_campaign%3Dvod%26utm_source%3Dmwc;S.browser_fallback_url=market%3A%2F%2Fdetails%3Fid%3Dtv.twitch.android.app;end;',
      );
    });

    it('builds a Fire deep link', () => {
      expect(
        buildDeepLinkURL({
          path: '/monstercat',
          referrer: 'ref',
          appStoreOpts: {
            os: OS.Fire,
            campaign: Location.VOD,
            callToActionIdentifier: 'a_button',
          },
        }),
      ).toEqual(
        'intent://m.twitch.tv/monstercat?referrer=ref#Intent;package=tv.twitch.android.viewer;scheme=twitch;S.market_referrer=utm_campaign%3Dvod%26utm_source%3Dmwc;end;',
      );
    });

    it('builds an iOS deep link', () => {
      expect(
        buildDeepLinkURL({
          path: '/monsterdog',
          referrer: 'ref',
          appStoreOpts: {
            os: OS.iOS,
            campaign: Location.VOD,
            callToActionIdentifier: 'b_button',
          },
        }),
      ).toEqual(`${IOS_INTERSTITIAL_BASE_URL}/monsterdog?c=vod-b_button`);
    });

    it('gives back an empty string for OSes without apps', () => {
      expect(
        buildDeepLinkURL({
          path: '/monsterfish',
          referrer: 'ref',
          appStoreOpts: {
            os: OS.Microsoft,
            campaign: Location.VOD,
            callToActionIdentifier: 'c_button',
          },
        }),
      ).toEqual('');
    });
  });
});
