import expect from 'expect';

import {
  estimateImageCountForViewport,
  ImageType,
} from 'mweb/common/utils/estimateImageCountForViewport';
import {
  BOX_ART_ASPECT_RATIO,
  THUMBNAIL_ASPECT_RATIO,
} from 'mweb/common/utils/imageUtils';

describe('estimateImageCountForViewport', () => {
  type Layout = 'games' | 'channels';
  type Estimates = { [K in Layout]: number };
  interface ViewportTestCase {
    device: string;
    clientWidth: number;
    clientHeight: number;
    estimates: Estimates;
  }

  const tests: ViewportTestCase[] = [
    {
      device: 'iPhone 5 Portrait',
      clientWidth: 320,
      clientHeight: 568,
      estimates: {
        games: 4,
        channels: 3,
      },
    },
    {
      device: 'iPhone 5 Landscape',
      clientWidth: 568,
      clientHeight: 320,
      estimates: {
        games: 8,
        channels: 4,
      },
    },
    {
      device: 'iPhone 6 Plus Portrait',
      clientWidth: 414,
      clientHeight: 736,
      estimates: {
        games: 6,
        channels: 3,
      },
    },
    {
      device: 'iPhone 6 Plus Landscape',
      clientWidth: 736,
      clientHeight: 414,
      estimates: {
        games: 8,
        channels: 4,
      },
    },
    {
      device: 'iPhone X Portrait',
      clientWidth: 375,
      clientHeight: 812,
      estimates: {
        games: 6,
        channels: 3,
      },
    },
    {
      device: 'iPhone X Landscape',
      clientWidth: 812,
      clientHeight: 375,
      estimates: {
        games: 8,
        channels: 6,
      },
    },
    {
      device: 'iPad Portrait',
      clientWidth: 768,
      clientHeight: 1024,
      estimates: {
        games: 16,
        channels: 8,
      },
    },
    {
      device: 'iPad Landscape',
      clientWidth: 1024,
      clientHeight: 768,
      estimates: {
        games: 8,
        channels: 9,
      },
    },
  ];

  interface LayoutTestCase {
    layout: Layout;
    imageType: ImageType;
    aspectRatio: number;
    heightFudge: number;
  }

  const layouts: LayoutTestCase[] = [
    {
      layout: 'games',
      imageType: ImageType.BoxArt,
      aspectRatio: BOX_ART_ASPECT_RATIO,
      heightFudge: 50,
    },
    {
      layout: 'channels',
      imageType: ImageType.Channel,
      aspectRatio: THUMBNAIL_ASPECT_RATIO,
      heightFudge: 67,
    },
  ];

  layouts.forEach(({ layout, imageType, aspectRatio, heightFudge }) => {
    describe(`${layout} layout`, () => {
      tests.forEach(({ device, clientWidth, clientHeight, estimates }) => {
        it(`has ${estimates[layout]} visibile images for ${device}`, () => {
          expect(
            estimateImageCountForViewport(
              aspectRatio,
              heightFudge,
              imageType,
              clientWidth,
              clientHeight,
            ),
          ).toEqual(estimates[layout]);
        });
      });
    });
  });
});
